@extends('backoffice.dashboard')

@section('title','تعديل صفحة')

@section('content')
<div class="container-fluid">
    <div class="block-card mb-4">
        <div class="block-card-header">
            <h2 class="widget-title"> تعديل الصفحة</h2>
            <div class="stroke-shape"></div>
        </div>

        <div class="block-card-body">
            <form action="{{ route('builder.update', $builder->id_builder) }}" method="POST" enctype="multipart/form-data" id="pageForm">
                @csrf
                @method('PUT')

                <!-- TITRES -->
                <div class="form-group mb-3">
                    <label>العنوان بالعربية</label>
                    <input type="text" class="form-control" name="titre_ar" value="{{ old('titre_ar', $builder->titre_ar) }}" 
                           maxlength="100" id="titre_ar">
                    <span class="field-error text-danger" id="error_titre_ar" style="display: none;"></span>
                </div>

                <div class="form-group mb-3">
                    <label>العنوان بالفرنسية</label>
                    <input type="text" class="form-control" name="titre_fr" value="{{ old('titre_fr', $builder->titre_fr) }}" 
                           maxlength="100" id="titre_fr">
                    <span class="field-error text-danger" id="error_titre_fr" style="display: none;"></span>
                </div>

                <div class="form-group mb-3">
                    <label>العنوان بالإنجليزية</label>
                    <input type="text" class="form-control" name="titre_en" value="{{ old('titre_en', $builder->titre_en) }}" 
                           maxlength="100" id="titre_en">
                    <span class="field-error text-danger" id="error_titre_en" style="display: none;"></span>
                </div>

                <hr>

                <h4 class="mb-3">المحتوى</h4>

                <!-- رسالة خطأ عامة للمحتوى -->
                <div class="alert alert-danger mb-3" id="general-error" style="display: none;">
                    <i class="ri-error-warning-line mr-2"></i>
                    <span id="general-error-message"></span>
                </div>

                <table class="table custom-table text-center" id="contenu-table">
                    <thead>
                        <tr>
                            <th>#</th>
                            <th>العرض (col)</th>
                            <th>النوع</th>
                            <th>المحتوى</th>
                            <th>إجراء</th>
                        </tr>
                    </thead>
                    <tbody id="contenu-body">
                        <!-- lignes dynamiques -->
                    </tbody>
                </table>

               

                <div class="block-card-footer mt-3 mb-3 text-center">
                    <button type="submit" class="btn bg-rgb-primary1 font-weight-medium">
                        <i class="ri-save-line mr-1"></i> حفظ
                    </button>
                    <a href="{{ route('builder.index') }}" class="btn bg-rgb-primary1 font-weight-medium">
                        <i class="ri-close-line mr-1"></i> إلغاء
                    </a>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- TEMPLATE LIGNE -->
<template id="row-template">
    <tr data-index="INDEX">
        <td class="row-number"></td>
        <input type="hidden" name="contenus[INDEX][id_builder_cont]" value="">
        <td>
            <select name="contenus[INDEX][col]" class="form-control col-select">
                <option value="col-xl-12">12</option>
                <option value="col-xl-8">8</option>
                <option value="col-xl-6">6</option>
                <option value="col-xl-4">4</option>
                <option value="col-xl-3">3</option>
                <option value="col-xl-2">2</option>
            </select>
        </td>
        <td>
            <select name="contenus[INDEX][nature]" class="form-control nature-select">
                <option value="">اختيار</option>
                <option value="image">صورة</option>
                <option value="video">فيديو</option>
                <option value="texte">نص</option>
            </select>
            <span class="field-error text-danger nature-error" style="display:none;"></span>
        </td>
        
        <td class="contenu-column">
            <!-- سيتم ملؤه ديناميكيًا -->
        </td>

        <td class="text-center">
            <button type="button" class="btn btn-success btn-sm me-1" onclick="addRow()">
                <i class="ri-add-line"></i>
            </button>
            <button type="button" class="btn btn-danger btn-sm" onclick="removeRow(this)">
                <i class="ri-delete-bin-line"></i>
            </button>
        </td>
    </tr>
</template>

<!-- SCRIPTS -->
<script src="https://cdn.jsdelivr.net/npm/tinymce@6/tinymce.min.js"></script>
<script>
let globalIndex = 0;
let isFormSubmitted = false;
const oldData = @json(old('contenus', []));

function initRow(row, data = null) {
    const index = row.getAttribute('data-index');
    const natureSelect = row.querySelector('.nature-select');
    const colSelect = row.querySelector('.col-select');
    const td = row.querySelector('.contenu-column');
    const natureError = row.querySelector('.nature-error');
    const hiddenInput = row.querySelector('input[name$="[id_builder_cont]"]');
    
    // استعادة القيم القديمة إن وجدت
    if (data) {
        if (data.id_builder_cont) hiddenInput.value = data.id_builder_cont;
        if (data.col) colSelect.value = data.col;
        if (data.nature) {
            natureSelect.value = data.nature;
            updateContentField(td, index, data);
        }
    }
    
    // إضافة حدث عند تغيير النوع
    natureSelect.addEventListener('change', function() {
        updateContentField(td, index, data);
        if (isFormSubmitted) {
            validateField(this, index, 'nature');
        }
    });
    
    // عند تغيير أي حقل في المحتوى، التحقق إذا تم النقر على الحفظ مسبقًا
    setTimeout(() => {
        const inputs = td.querySelectorAll('input, textarea');
        inputs.forEach(input => {
            input.addEventListener('input', function() {
                if (isFormSubmitted) {
                    validateContentField(row, index);
                }
            });
            input.addEventListener('change', function() {
                if (isFormSubmitted) {
                    validateContentField(row, index);
                }
            });
        });
    }, 300);
}

function updateContentField(td, index, oldData = null) {
    const natureSelect = td.closest('tr').querySelector('.nature-select');
    const nature = natureSelect.value;
    td.innerHTML = '';

    if (nature === 'image') {
        let preview = '';
        let existingImage = '';
        
        if (oldData && oldData.image) {
            // إذا كانت صورة موجودة من قاعدة البيانات
            if (oldData.image.startsWith('data:')) {
                preview = `<img src="${oldData.image}" width="120" class="mt-2 mb-2 d-block"><small class="text-muted">الصورة الحالية</small>`;
            } else {
                // إذا كان لدينا اسم ملف أو معرف
                existingImage = `<input type="hidden" name="contenus[${index}][existing_image]" value="${oldData.image}">`;
            }
        }
        
        td.innerHTML = `
            ${preview}
            ${existingImage}
            <input type="file" name="contenus[${index}][image]" class="form-control file-input" accept="image/*">
            <small class="text-muted d-block mt-1">اتركه فارغاً للحفاظ على الصورة الحالية</small>
            <span class="field-error text-danger content-error" id="error_content_${index}" style="display:none;"></span>
        `;
        
        const fileInput = td.querySelector('.file-input');
        fileInput.addEventListener('change', function() {
            if (isFormSubmitted) {
                validateContentField(td.closest('tr'), index);
            }
        });
    }
    else if (nature === 'video') {
        const oldUrl = oldData ? oldData.url_video : '';
        td.innerHTML = `
            <input type="text" name="contenus[${index}][url_video]" class="form-control video-input" 
                   placeholder="رابط الفيديو (YouTube)" value="${oldUrl}">
            <span class="field-error text-danger content-error" id="error_content_${index}" style="display:none;"></span>
        `;
        
        const videoInput = td.querySelector('.video-input');
        videoInput.addEventListener('input', function() {
            if (isFormSubmitted) {
                validateContentField(td.closest('tr'), index);
            }
        });
    }
    else if (nature === 'texte') {
        const oldTextAr = oldData ? oldData.text_ar : '';
        const oldTextFr = oldData ? oldData.text_fr : '';
        const oldTextEn = oldData ? oldData.text_en : '';
        
        td.innerHTML = `
            <div class="mb-2">
                <small>العربية</small>
                <textarea name="contenus[${index}][text_ar]" rows="3" 
                          class="form-control tinymce-text" id="text_ar_${index}">${oldTextAr}</textarea>
            </div>
            <div class="mb-2">
                <small>الفرنسية</small>
                <textarea name="contenus[${index}][text_fr]" rows="3" 
                          class="form-control tinymce-text" id="text_fr_${index}">${oldTextFr}</textarea>
            </div>
            <div class="mb-2">
                <small>الإنجليزية</small>
                <textarea name="contenus[${index}][text_en]" rows="3" 
                          class="form-control tinymce-text" id="text_en_${index}">${oldTextEn}</textarea>
            </div>
            <span class="field-error text-danger content-error" id="error_content_${index}" style="display:none;"></span>
        `;
        
        // تهيئة TinyMCE
        setTimeout(() => {
            tinymce.init({
                selector: `#text_ar_${index}, #text_fr_${index}, #text_en_${index}`,
                height: 150,
                plugins: 'advlist autolink lists link image preview code',
                toolbar: 'undo redo | bold italic | bullist numlist | alignleft aligncenter alignright | code',
                setup: function(editor) {
                    editor.on('change', function() {
                        if (isFormSubmitted) {
                            validateContentField(td.closest('tr'), index);
                        }
                    });
                }
            });
        }, 100);
    }
}

function addRow(data = null) {
    const tpl = document.querySelector('#row-template').innerHTML;
    const newTpl = tpl.replace(/INDEX/g, globalIndex);
    const table = document.querySelector('#contenu-body');
    table.insertAdjacentHTML('beforeend', newTpl);
    
    const newRow = table.lastElementChild;
    initRow(newRow, data);
    renumber();
    globalIndex++;
}

function removeRow(btn) {
    btn.closest('tr').remove();
    renumber();
}

function renumber() {
    document.querySelectorAll('#contenu-body tr').forEach((row, i) => {
        row.querySelector('.row-number').innerText = i + 1;
    });
}

function validateField(element, index, fieldType) {
    const errorSpan = fieldType === 'nature' 
        ? element.nextElementSibling 
        : document.querySelector(`#error_${fieldType}_${index}`);
    
    if (errorSpan) {
        let isValid = true;
        let message = '';
        
        if (fieldType === 'nature') {
            if (!element.value) {
                message = 'يرجى اختيار نوع المحتوى';
                isValid = false;
            }
        }
        
        if (!isValid) {
            errorSpan.textContent = message;
            errorSpan.style.display = 'block';
            element.classList.add('is-invalid');
        } else {
            errorSpan.style.display = 'none';
            element.classList.remove('is-invalid');
        }
        
        return isValid;
    }
    return true;
}

function validateContentField(row, index) {
    const natureSelect = row.querySelector('.nature-select');
    const nature = natureSelect.value;
    let isValid = true;
    let message = '';
    const errorSpan = row.querySelector('.content-error');
    
    if (nature === 'image') {
        // الصورة ليست إلزامية في التعديل إذا كانت موجودة مسبقاً
        const fileInput = row.querySelector('.file-input');
        const existingImage = row.querySelector('input[name$="[existing_image]"]') || row.querySelector('img');
        
        if (!existingImage && (!fileInput || fileInput.files.length === 0)) {
            message = 'يرجى اختيار صورة أو يوجد صورة حالية';
            isValid = false;
        }
        
        if (errorSpan) {
            errorSpan.textContent = message;
            errorSpan.style.display = isValid ? 'none' : 'block';
            if (fileInput) fileInput.classList.toggle('is-invalid', !isValid);
        }
    }
    else if (nature === 'video') {
        const videoInput = row.querySelector('.video-input');
        
        if (!videoInput || !videoInput.value.trim()) {
            message = 'يرجى إدخال رابط الفيديو';
            isValid = false;
        } else if (!isValidYoutubeUrl(videoInput.value.trim())) {
            message = 'يرجى إدخال رابط YouTube صحيح';
            isValid = false;
        }
        
        if (errorSpan) {
            errorSpan.textContent = message;
            errorSpan.style.display = isValid ? 'none' : 'block';
            if (videoInput) videoInput.classList.toggle('is-invalid', !isValid);
        }
    }
else if (nature === 'texte') {
    const textareas = row.querySelectorAll('.tinymce-text');
    let hasEmpty = false;

    textareas.forEach(textarea => {
        let contentHtml = '';
        let textOnly = '';

        const editor = tinymce.get(textarea.id);
        if (editor) {
            contentHtml = editor.getContent().trim(); // HTML كامل
            textOnly = editor.getContent({ format: 'text' }).trim(); // النص فقط
        } else {
            contentHtml = textarea.value.trim();
            // ازالة الوسوم للحصول على نص فقط
            textOnly = textarea.value.replace(/<[^>]*>/g, '').trim();
        }

        // قبول الحقل إذا كان هناك HTML (مثلاً iframe) أو نص
        // أي: يعتبر صالحاً إذا كان هناك نص أو كان HTML يحتوي iframe أو أي علامة غير فارغة
        if (!contentHtml || (textOnly.length === 0 && !/iframe/i.test(contentHtml))) {
            hasEmpty = true;
        }
    });

    if (hasEmpty) {
        message = 'جميع النصوص مطلوبة (أو ضع iframe/محتوى HTML صالح)';
        isValid = false;
    }

    if (errorSpan) {
        errorSpan.textContent = message;
        errorSpan.style.display = isValid ? 'none' : 'block';
        textareas.forEach(textarea => {
            textarea.classList.toggle('is-invalid', !isValid);
        });
    }
}

    
    return isValid;
}

function isValidYoutubeUrl(url) {
    const regex = /^(https?:\/\/)?(www\.)?(youtube\.com\/watch\?v=|youtu\.be\/)[\w-]{11}/;
    return regex.test(url);
}

// التحقق من صحة العناوين (فقط عند النقر على الحفظ)
function validateTitles() {
    let isValid = true;
    const titles = ['titre_ar', 'titre_fr', 'titre_en'];
    
    titles.forEach(title => {
        const input = document.getElementById(title);
        const error = document.getElementById(`error_${title}`);
        
        if (!input.value.trim()) {
            error.textContent = 'هذا الحقل مطلوب';
            error.style.display = 'block';
            input.classList.add('is-invalid');
            isValid = false;
        } else if (input.value.length > 100) {
            error.textContent = 'الحد الأقصى 100 حرف';
            error.style.display = 'block';
            input.classList.add('is-invalid');
            isValid = false;
        } else {
            error.style.display = 'none';
            input.classList.remove('is-invalid');
        }
    });
    
    return isValid;
}

// التحقق من صحة جميع المحتويات (فقط عند النقر على الحفظ)
function validateAllContents() {
    let isValid = true;
    const rows = document.querySelectorAll('#contenu-body tr');
    const generalError = document.getElementById('general-error');
    const generalErrorMessage = document.getElementById('general-error-message');
    
    // إخفاء الخطأ العام
    generalError.style.display = 'none';
    
    if (rows.length === 0) {
        generalErrorMessage.textContent = 'يجب إضافة محتوى واحد على الأقل';
        generalError.style.display = 'block';
        return false;
    }
    
    rows.forEach((row, index) => {
        const natureSelect = row.querySelector('.nature-select');
        const natureValid = validateField(natureSelect, index, 'nature');
        const contentValid = validateContentField(row, index);
        
        if (!natureValid || !contentValid) {
            isValid = false;
        }
    });
    
    if (!isValid) {
        generalErrorMessage.textContent = 'يوجد أخطاء في بعض الحقول. الرجاء التحقق من جميع الصفوف.';
        generalError.style.display = 'block';
    }
    
    return isValid;
}

// حدث عند تقديم النموذج
document.getElementById('pageForm').addEventListener('submit', function(e) {
    e.preventDefault();
    
    // تعيين المتغير للإشارة إلى أن النموذج تمت محاولة إرساله
    isFormSubmitted = true;
    
    const titlesValid = validateTitles();
    const contentsValid = validateAllContents();
    
    if (titlesValid && contentsValid) {
        // إذا كان كل شيء صحيحاً، نرسل النموذج
        this.submit();
    } else {
        // التمرير إلى أول خطأ
        const firstError = document.querySelector('.is-invalid');
        if (firstError) {
            firstError.scrollIntoView({ behavior: 'smooth', block: 'center' });
            firstError.focus();
        }
    }
});

// أحداث للتحقق الفوري بعد النقر الأول على الحفظ
document.querySelectorAll('#titre_ar, #titre_fr, #titre_en').forEach(input => {
    input.addEventListener('input', function() {
        if (isFormSubmitted) {
            validateTitles();
        }
    });
});

// تهيئة الصفحة
document.addEventListener('DOMContentLoaded', () => {
    // تحديد ما إذا كنا نستخدم البيانات القديمة أم بيانات النموذج
    let initialData = [];
    
    if (oldData && oldData.length > 0) {
        // استخدم البيانات القديمة إذا كانت موجودة (بعد إعادة التوجيه مع أخطاء)
        initialData = oldData;
    } else {
        // استخدم البيانات الحالية من قاعدة البيانات
        initialData = @json($builder->contenus->map(function($contenu) {
            $data = $contenu->toArray();
            if ($contenu->nature === 'image' && $contenu->image) {
                $data['image'] = 'data:image/jpeg;base64,' . base64_encode($contenu->image);
            }
            return $data;
        }));
    }
    
    // تحميل البيانات
    if (initialData.length > 0) {
        initialData.forEach(data => {
            addRow(data);
        });
    } else {
        // إضافة صف افتراضي
        addRow();
    }
});
</script>

<!-- STYLES -->
<style>
.is-invalid {
    border: 1px solid #dc3545 !important;
}
.field-error {
    font-size: 13px;
    margin-top: 4px;
    display: none;
}
#general-error {
    display: none;
    animation: fadeIn 0.3s ease-in-out;
}
@keyframes fadeIn {
    from { opacity: 0; }
    to { opacity: 1; }
}
.custom-table {
    width: 100%;
    border-collapse: separate;
    border-spacing: 0;
    border-radius: 8px;
    overflow: hidden;
    background-color: #fff;
    box-shadow: 0 1px 4px rgba(0,0,0,0.1);
}
.custom-table th, .custom-table td {
    vertical-align: middle;
    text-align: center;
    padding: 12px 8px;
    border-bottom: 1px solid #e9ecef;
}
.btn-sm { 
    padding: 6px 12px; 
    font-size: 14px;
    display: inline-flex;
    align-items: center;
    justify-content: center;
    min-width: 36px;
    min-height: 36px;
}
.btn-sm i {
    font-size: 16px;
    line-height: 1;
}
.me-1 { 
    margin-right: 8px !important; 
}
.custom-table td.text-center {
    white-space: nowrap;
}
.block-card-footer .btn {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 10px 20px;
    min-width: 100px;
}
.block-card-footer .btn i {
    font-size: 18px;
}
.tox-tinymce {
    border: 1px solid #ced4da !important;
    border-radius: 0.375rem;
}
.tox-tinymce.is-invalid {
    border: 1px solid #dc3545 !important;
}
.nature-error, .content-error {
    display: none;
}
</style>

@endsection