<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Notifications\Notification;

class StatutDocumentNotification extends Notification
{
    use Queueable;

    protected $docList;
    protected $type;           // normalized type: accepted|rejected|incomplete|deferred|status_changed
    protected $params;         // assoc array for translation params
    protected $referenceId;
    protected $translationKey; // explicit key (optional)

    /**
     * $docList: DocumentsFinancementList model
     * $type: normalized type name (accepted|rejected|incomplete|deferred|status_changed)
     * $params: array of translation params (e.g. ['reason' => '...'])
     * $referenceId: optional related id (eg. $docList->id_doc_list)
     * $translationKey: optional explicit translation key to use instead of auto-building one
     */
    public function __construct($docList, string $type, array $params = [], $referenceId = null, ?string $translationKey = null)
    {
        $this->docList = $docList;
        $this->type = $type;
        $this->params = $params;
        $this->referenceId = $referenceId;
        $this->translationKey = $translationKey;
    }

    public function via($notifiable)
    {
        return ['database'];
    }

    public function toDatabase($notifiable)
    {
        // dossier number (same logic as before)
        $createdAt = $this->docList->created_at ?? now();
        $numDossier = $createdAt->format('Y') . '_' .
                      ($this->docList->user->cin ?? '00000000') . '_' .
                      str_pad($this->docList->id_doc_list, 3, '0', STR_PAD_LEFT) . '_00';

        // If an explicit translation key was provided in the constructor, use it;
        // otherwise build one from the normalized type.
        $translationKey = $this->translationKey ?? "site.notifications.documents_formulaire.{$this->type}";

        return [
            // EXACT fields your blade expects:
            'translation_key' => $translationKey,
            'translation_params' => $this->params,
            'num_dossier' => $numDossier,
            'date' => now()->toDateTimeString(),
            'reference_id' => $this->referenceId,
        ];
    }

    public function toArray($notifiable)
    {
        return [
            'type' => $this->type,
            'params' => $this->params,
            'reference_id' => $this->referenceId,
        ];
    }
}
