<?php

namespace App\Http\Controllers;

use App\Models\Slider;
use App\Models\Utilisateur;
use App\Rules\ValidLink; // إضافة القاعدة المخصصة
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;

class SlidersController extends Controller
{
   // controller (e.g. Backoffice\SliderController)

public function index(Request $request)
{
    $search = trim($request->get('search'));
    $words = $search ? preg_split('/\s+/', $search) : [];

    $sliders = Slider::with('utilisateur')
        ->when($words, function ($query) use ($words) {
            foreach ($words as $word) {
                $word = str_replace(['\\', '%', '_'], ['\\\\', '\\%', '\\_'], $word);
                $query->where(function ($q) use ($word) {
                    $q->where('titre_ar', 'LIKE', "%{$word}%")
                      ->orWhere('titre_fr', 'LIKE', "%{$word}%")
                      ->orWhere('titre_en', 'LIKE', "%{$word}%");
                });
            }
        })
        ->orderBy('created_at', 'desc')
        ->paginate(10)
        ->appends(['search' => $search]);

    // helper to compute imageSrc
    $computeImageSrc = function ($stored) {
        if (empty($stored)) return null;

        // if it's already a full URL
        if (is_string($stored) && filter_var($stored, FILTER_VALIDATE_URL)) {
            return $stored;
        }

        // if binary blob stored in DB
        if (!is_string($stored)) {
            try {
                return 'data:image/jpeg;base64,' . base64_encode($stored);
            } catch (\Throwable $e) {
                // fallthrough to file checks
            }
        }

        // candidate filesystem locations / web exposures
        $candidates = [
            // as stored (maybe absolute or relative)
            (string)$stored,
            // server root prefix you use
            '/www/mjs' . (strpos((string)$stored, '/') === 0 ? (string)$stored : '/' . ltrim((string)$stored, '/')),
            // shared_uploads/images with basename
            '/www/mjs/shared_uploads/images/' . basename((string)$stored),
            // another possible absolute path
            '/home/preprov/www/mjs/shared_uploads/images/' . basename((string)$stored),
            // public path (web-accessible)
            public_path(ltrim((string)$stored, '/')),
            // storage fallback
            storage_path('app/' . ltrim((string)$stored, '/')),
        ];

        $seen = [];
        foreach ($candidates as $path) {
            if (!$path) continue;
            $path = preg_replace('#^file://#', '', $path);
            if (isset($seen[$path])) continue;
            $seen[$path] = true;

            if (@file_exists($path) && @is_file($path) && @is_readable($path)) {
                $mime = @mime_content_type($path) ?: 'image/jpeg';
                $content = @file_get_contents($path);
                if ($content !== false) {
                    return "data:{$mime};base64," . base64_encode($content);
                }
            }
        }

        // if stored starts with a slash and may be web-accessible, expose as URL
        if (is_string($stored) && strpos($stored, '/') === 0) {
            return url(ltrim($stored, '/'));
        }

        // last resort: expose basename inside shared_uploads/images as URL
        return url('shared_uploads/images/' . basename((string)$stored));
    };

    // attach imageSrc
    foreach ($sliders as $s) {
        $s->imageSrc = $computeImageSrc($s->image);
    }

    return view('backoffice.sliders.index', compact('sliders'));
}

public function edit($id)
{
    $slider = Slider::findOrFail($id);

    // the same helper as above (kept inline for minimal changes)
    $computeImageSrc = function ($stored) {
        if (empty($stored)) return null;
        if (is_string($stored) && filter_var($stored, FILTER_VALIDATE_URL)) return $stored;
        if (!is_string($stored)) {
            try { return 'data:image/jpeg;base64,' . base64_encode($stored); } catch (\Throwable $e) {}
        }
        $candidates = [
            (string)$stored,
            '/www/mjs' . (strpos((string)$stored, '/') === 0 ? (string)$stored : '/' . ltrim((string)$stored, '/')),
            '/www/mjs/shared_uploads/images/' . basename((string)$stored),
            '/home/preprov/www/mjs/shared_uploads/images/' . basename((string)$stored),
            public_path(ltrim((string)$stored, '/')),
            storage_path('app/' . ltrim((string)$stored, '/')),
        ];
        $seen = [];
        foreach ($candidates as $path) {
            if (!$path) continue;
            $path = preg_replace('#^file://#', '', $path);
            if (isset($seen[$path])) continue;
            $seen[$path] = true;
            if (@file_exists($path) && @is_file($path) && @is_readable($path)) {
                $mime = @mime_content_type($path) ?: 'image/jpeg';
                $content = @file_get_contents($path);
                if ($content !== false) {
                    return "data:{$mime};base64," . base64_encode($content);
                }
            }
        }
        if (is_string($stored) && strpos($stored, '/') === 0) return url(ltrim($stored, '/'));
        return url('shared_uploads/images/' . basename((string)$stored));
    };

    $slider->imageSrc = $computeImageSrc($slider->image);

    return view('backoffice.sliders.edit', compact('slider'));
}

    public function create()
    {
        return view('backoffice.sliders.create');
    }

public function store(Request $request) 
{
    $validator = Validator::make($request->all(), [
        'titre_ar' => 'required|string|max:100',
        'titre_fr' => 'required|string|max:100',
        'titre_en' => 'required|string|max:100',
        'soustitre_ar' => 'nullable|string|max:100',
        'soustitre_fr' => 'nullable|string|max:100',
        'soustitre_en' => 'nullable|string|max:100',

        'titre_bouton_ar' => 'nullable|string|max:100',
        'lien_bouton_ar' => ['nullable', 'max:255', new ValidLink],
        'titre_bouton_fr' => 'nullable|string|max:100',
        'lien_bouton_fr' => ['nullable', 'max:255', new ValidLink],
        'titre_bouton_en' => 'nullable|string|max:100',
        'lien_bouton_en' => ['nullable', 'max:255', new ValidLink],

        'titre_bouton2_ar' => 'nullable|string|max:100',
        'lien_bouton2_ar' => ['nullable', 'max:255', new ValidLink],
        'titre_bouton2_fr' => 'nullable|string|max:100',
        'lien_bouton2_fr' => ['nullable', 'max:255', new ValidLink],
        'titre_bouton2_en' => 'nullable|string|max:100',
        'lien_bouton2_en' => ['nullable', 'max:255', new ValidLink],

        'image' => 'required|image|mimes:jpeg,png,jpg,webp|max:2048',
    ], [
        'titre_ar.required' => 'الرجاء إدخال العنوان بالعربية',
        'titre_fr.required' => 'الرجاء إدخال العنوان بالفرنسية',
        'titre_en.required' => 'الرجاء إدخال العنوان بالإنجليزية',
        'image.required' => 'الرجاء اختيار صورة',
        'image.image' => 'الملف يجب أن يكون صورة',
        'image.mimes' => 'نوع الصورة يجب أن يكون jpeg, png, jpg أو webp',
        'image.max' => 'حجم الصورة يجب ألا يتجاوز 2 ميغابايت',
    ]);

    if ($validator->fails()) {
        return redirect()->back()->withErrors($validator)->withInput();
    }

    // Candidate directories to try for saving uploaded images (ordered).
    $candidates = [
        '/www/mjs/shared_uploads/images',
        '/home/preprov/www/mjs/shared_uploads/images',
        base_path('shared_uploads/images'),
        base_path('../shared_uploads/images'),
        public_path('../../shared_uploads/images'),
    ];

    $usable = null;
    $tried = [];
    $imagePath = null;

    if ($request->hasFile('image')) {
        $file = $request->file('image');

        foreach ($candidates as $cand) {
            $rp = @realpath($cand);
            $tried[$cand] = [
                'realpath' => $rp ?: null,
                'exists' => @file_exists($cand) ? 'yes' : 'no',
                'is_dir' => @is_dir($cand) ? 'yes' : 'no',
                'is_writable' => @is_writable($cand) ? 'yes' : 'no'
            ];
            // Choose the candidate only if realpath exists, it's a directory, and writable.
            if ($rp && is_dir($rp) && is_writable($rp)) {
                $usable = $rp;
                break;
            }
        }

        if (!$usable) {
            // useful diagnostics for logs and user
            $diag = [
                'cwd' => @getcwd(),
                'open_basedir' => ini_get('open_basedir') ?: '(not set)',
                'tried' => $tried,
                'get_current_user' => @get_current_user(),
            ];
            \Log::warning('No usable upload folder found for slider image upload.', $diag);

            $msg = "Upload folder not accessible by PHP. Checked these candidates: ";
            foreach ($tried as $c => $info) {
                $msg .= "[{$c} -> realpath=" . ($info['realpath'] ?? 'null') . ", exists={$info['exists']}, is_dir={$info['is_dir']}, writable={$info['is_writable']}] ";
            }
            return back()->withInput()->withErrors(['image' => $msg]);
        }

        // sanitize original filename (keep original name where possible)
        $originalName = $file->getClientOriginalName();
        $originalName = basename($originalName); // strip any path
        $originalName = preg_replace('/[\0\/\\\\<>:"|?*\x00-\x1F]+/u', '-', $originalName);
        if ($originalName === '' || $originalName === '.' || $originalName === '..') {
            return back()->withInput()->withErrors(['image' => 'اسم الملف الأصلي غير صالح.']);
        }

        $targetPath = rtrim($usable, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $originalName;

        // avoid overwriting: append timestamp if exists
        if (file_exists($targetPath)) {
            $dotPos = strrpos($originalName, '.');
            if ($dotPos !== false) {
                $base = substr($originalName, 0, $dotPos);
                $ext  = substr($originalName, $dotPos);
            } else {
                $base = $originalName;
                $ext = '';
            }
            $originalName = $base . '-' . time() . $ext;
            $targetPath = rtrim($usable, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $originalName;
        }

        try {
            // move the uploaded file to the chosen directory (uses move_uploaded_file internally)
            $file->move($usable, $originalName);

            // store the absolute path in the DB (example: /www/mjs/shared_uploads/images/filename.png)
            $imagePath = $targetPath;
        } catch (\Throwable $e) {
            \Log::error('Slider image move failed: ' . $e->getMessage(), ['exception' => $e, 'usable' => $usable]);
            return back()->withInput()->withErrors(['image' => 'فشل حفظ الصورة على الخادم. (راجع سجلات الخادم).']);
        }
    }

    Slider::create([
        'titre_ar' => $request->titre_ar,
        'titre_fr' => $request->titre_fr,
        'titre_en' => $request->titre_en,
        'soustitre_ar' => $request->soustitre_ar,
        'soustitre_fr' => $request->soustitre_fr,
        'soustitre_en' => $request->soustitre_en,

        'titre_bouton_ar' => $request->titre_bouton_ar,
        'lien_bouton_ar' => $request->lien_bouton_ar,
        'titre_bouton_fr' => $request->titre_bouton_fr,
        'lien_bouton_fr' => $request->lien_bouton_fr,
        'titre_bouton_en' => $request->titre_bouton_en,
        'lien_bouton_en' => $request->lien_bouton_en,

        'titre_bouton2_ar' => $request->titre_bouton2_ar,
        'lien_bouton2_ar' => $request->lien_bouton2_ar,
        'titre_bouton2_fr' => $request->titre_bouton2_fr,
        'lien_bouton2_fr' => $request->lien_bouton2_fr,
        'titre_bouton2_en' => $request->titre_bouton2_en,
        'lien_bouton2_en' => $request->lien_bouton2_en,

        // store the path instead of binary content
        'image' => $imagePath,
        'id_user' => Auth::id(),
    ]);

    return redirect()->route('sliders.index')->with('success', 'تمت إضافة الصورة بنجاح');
}


 /*   public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'titre_ar' => 'required|string|max:100',
            'titre_fr' => 'required|string|max:100',
            'titre_en' => 'required|string|max:100',
            'soustitre_ar' => 'nullable|string|max:100',
            'soustitre_fr' => 'nullable|string|max:100',
            'soustitre_en' => 'nullable|string|max:100',

            'titre_bouton_ar' => 'nullable|string|max:100',
            'lien_bouton_ar' => ['nullable', 'max:255', new ValidLink],
            'titre_bouton_fr' => 'nullable|string|max:100',
            'lien_bouton_fr' => ['nullable', 'max:255', new ValidLink],
            'titre_bouton_en' => 'nullable|string|max:100',
            'lien_bouton_en' => ['nullable', 'max:255', new ValidLink],

            'titre_bouton2_ar' => 'nullable|string|max:100',
            'lien_bouton2_ar' => ['nullable', 'max:255', new ValidLink],
            'titre_bouton2_fr' => 'nullable|string|max:100',
            'lien_bouton2_fr' => ['nullable', 'max:255', new ValidLink],
            'titre_bouton2_en' => 'nullable|string|max:100',
            'lien_bouton2_en' => ['nullable', 'max:255', new ValidLink],

            'image' => 'required|image|mimes:jpeg,png,jpg,webp|max:2048',
        ], [
            'titre_ar.required' => 'الرجاء إدخال العنوان بالعربية',
            'titre_fr.required' => 'الرجاء إدخال العنوان بالفرنسية',
            'titre_en.required' => 'الرجاء إدخال العنوان بالإنجليزية',
            'image.required' => 'الرجاء اختيار صورة',
            'image.image' => 'الملف يجب أن يكون صورة',
            'image.mimes' => 'نوع الصورة يجب أن يكون jpeg, png, jpg أو webp',
            'image.max' => 'حجم الصورة يجب ألا يتجاوز 2 ميغابايت',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $imageData = null;
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $imageData = file_get_contents($image->getRealPath());
        }

        Slider::create([
            'titre_ar' => $request->titre_ar,
            'titre_fr' => $request->titre_fr,
            'titre_en' => $request->titre_en,
            'soustitre_ar' => $request->soustitre_ar,
            'soustitre_fr' => $request->soustitre_fr,
            'soustitre_en' => $request->soustitre_en,

            'titre_bouton_ar' => $request->titre_bouton_ar,
            'lien_bouton_ar' => $request->lien_bouton_ar,
            'titre_bouton_fr' => $request->titre_bouton_fr,
            'lien_bouton_fr' => $request->lien_bouton_fr,
            'titre_bouton_en' => $request->titre_bouton_en,
            'lien_bouton_en' => $request->lien_bouton_en,

            'titre_bouton2_ar' => $request->titre_bouton2_ar,
            'lien_bouton2_ar' => $request->lien_bouton2_ar,
            'titre_bouton2_fr' => $request->titre_bouton2_fr,
            'lien_bouton2_fr' => $request->lien_bouton2_fr,
            'titre_bouton2_en' => $request->titre_bouton2_en,
            'lien_bouton2_en' => $request->lien_bouton2_en,

            'image' => $imageData,
            'id_user' => Auth::id(),
        ]);

        return redirect()->route('sliders.index')->with('success', 'تمت إضافة الصورة بنجاح');
    }
*/
 
public function update(Request $request, $id) 
{
    $slider = Slider::findOrFail($id);

    $validator = Validator::make($request->all(), [
        'titre_ar' => 'required|string|max:100',
        'titre_fr' => 'required|string|max:100',
        'titre_en' => 'required|string|max:100',
        'soustitre_ar' => 'nullable|string|max:100',
        'soustitre_fr' => 'nullable|string|max:100',
        'soustitre_en' => 'nullable|string|max:100',

        'titre_bouton_ar' => 'nullable|string|max:100',
        'lien_bouton_ar' => ['nullable', 'max:255', new ValidLink],
        'titre_bouton_fr' => 'nullable|string|max:100',
        'lien_bouton_fr' => ['nullable', 'max:255', new ValidLink],
        'titre_bouton_en' => 'nullable|string|max:100',
        'lien_bouton_en' => ['nullable', 'max:255', new ValidLink],

        'titre_bouton2_ar' => 'nullable|string|max:100',
        'lien_bouton2_ar' => ['nullable', 'max:255', new ValidLink],
        'titre_bouton2_fr' => 'nullable|string|max:100',
        'lien_bouton2_fr' => ['nullable', 'max:255', new ValidLink],
        'titre_bouton2_en' => 'nullable|string|max:100',
        'lien_bouton2_en' => ['nullable', 'max:255', new ValidLink],

        'image' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:2048',
    ], [
        'titre_ar.required' => 'الرجاء إدخال العنوان بالعربية',
        'titre_fr.required' => 'الرجاء إدخال العنوان بالفرنسية',
        'titre_en.required' => 'الرجاء إدخال العنوان بالإنجليزية',
        'image.image' => 'الملف يجب أن يكون صورة',
        'image.mimes' => 'نوع الصورة يجب أن يكون jpeg, png, jpg أو webp',
        'image.max' => 'حجم الصورة يجب ألا يتجاوز 2 ميغابايت',
    ]);

    if ($validator->fails()) {
        return redirect()->back()->withErrors($validator)->withInput();
    }

    // Candidate directories to try for saving uploaded images (ordered).
    $candidates = [
        '/www/mjs/shared_uploads/images',
        '/home/preprov/www/mjs/shared_uploads/images',
        base_path('shared_uploads/images'),
        base_path('../shared_uploads/images'),
        public_path('../../shared_uploads/images'),
    ];

    $usable = null;
    $tried = [];
    $newImagePath = null;

    if ($request->hasFile('image')) {
        $file = $request->file('image');

        foreach ($candidates as $cand) {
            $rp = @realpath($cand);
            $tried[$cand] = [
                'realpath' => $rp ?: null,
                'exists' => @file_exists($cand) ? 'yes' : 'no',
                'is_dir' => @is_dir($cand) ? 'yes' : 'no',
                'is_writable' => @is_writable($cand) ? 'yes' : 'no'
            ];
            if ($rp && is_dir($rp) && is_writable($rp)) {
                $usable = $rp;
                break;
            }
        }

        if (!$usable) {
            $diag = [
                'cwd' => @getcwd(),
                'open_basedir' => ini_get('open_basedir') ?: '(not set)',
                'tried' => $tried,
                'get_current_user' => @get_current_user(),
            ];
            \Log::warning('No usable upload folder found for slider image update.', $diag);

            $msg = "Upload folder not accessible by PHP. Checked these candidates: ";
            foreach ($tried as $c => $info) {
                $msg .= "[{$c} -> realpath=" . ($info['realpath'] ?? 'null') . ", exists={$info['exists']}, is_dir={$info['is_dir']}, writable={$info['is_writable']}] ";
            }
            return back()->withInput()->withErrors(['image' => $msg]);
        }

        // sanitize original filename (keep original name where possible)
        $originalName = $file->getClientOriginalName();
        $originalName = basename($originalName);
        $originalName = preg_replace('/[\0\/\\\\<>:"|?*\x00-\x1F]+/u', '-', $originalName);
        if ($originalName === '' || $originalName === '.' || $originalName === '..') {
            return back()->withInput()->withErrors(['image' => 'اسم الملف الأصلي غير صالح.']);
        }

        $targetPath = rtrim($usable, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $originalName;

        // avoid overwriting: append timestamp if exists
        if (file_exists($targetPath)) {
            $dotPos = strrpos($originalName, '.');
            if ($dotPos !== false) {
                $base = substr($originalName, 0, $dotPos);
                $ext  = substr($originalName, $dotPos);
            } else {
                $base = $originalName;
                $ext = '';
            }
            $originalName = $base . '-' . time() . $ext;
            $targetPath = rtrim($usable, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $originalName;
        }

        try {
            // move uploaded file
            $file->move($usable, $originalName);
            $newImagePath = $targetPath;
        } catch (\Throwable $e) {
            \Log::error('Slider image move failed during update: ' . $e->getMessage(), ['exception' => $e, 'usable' => $usable]);
            return back()->withInput()->withErrors(['image' => 'فشل حفظ الصورة على الخادم. (راجع سجلات الخادم).']);
        }

        // delete old image file if it looks like a filesystem path and exists
        try {
            if (!empty($slider->image) && is_string($slider->image)) {
                $oldPath = $slider->image;
                // try several representations to maximize chance to remove the correct file
                $possibleOldPaths = [
                    $oldPath,
                    // if oldPath is absolute but missing server root, try prefixing
                    '/www/mjs' . (strpos($oldPath, '/') === 0 ? $oldPath : '/' . ltrim($oldPath, '/')),
                    // basename in shared folder
                    '/www/mjs/shared_uploads/images/' . basename($oldPath),
                    // public path fallback
                    public_path(ltrim($oldPath, '/')),
                ];
                foreach ($possibleOldPaths as $p) {
                    if ($p && @file_exists($p) && @is_file($p)) {
                        @unlink($p);
                        // don't break — attempt to remove duplicates if any
                    }
                }
            }
        } catch (\Throwable $e) {
            // Log but don't block update if deletion fails
            \Log::warning('Failed deleting old slider image during update: ' . $e->getMessage(), ['slider_id' => $slider->id]);
        }
    }

    // prepare data to update
    $updateData = [
        'titre_ar' => $request->titre_ar,
        'titre_fr' => $request->titre_fr,
        'titre_en' => $request->titre_en,
        'soustitre_ar' => $request->soustitre_ar,
        'soustitre_fr' => $request->soustitre_fr,
        'soustitre_en' => $request->soustitre_en,

        'titre_bouton_ar' => $request->titre_bouton_ar,
        'lien_bouton_ar' => $request->lien_bouton_ar,
        'titre_bouton_fr' => $request->titre_bouton_fr,
        'lien_bouton_fr' => $request->lien_bouton_fr,
        'titre_bouton_en' => $request->titre_bouton_en,
        'lien_bouton_en' => $request->lien_bouton_en,

        'titre_bouton2_ar' => $request->titre_bouton2_ar,
        'lien_bouton2_ar' => $request->lien_bouton2_ar,
        'titre_bouton2_fr' => $request->titre_bouton2_fr,
        'lien_bouton2_fr' => $request->lien_bouton2_fr,
        'titre_bouton2_en' => $request->titre_bouton2_en,
        'lien_bouton2_en' => $request->lien_bouton2_en,
        'id_user' => Auth::id(),
    ];

    // if a new image was uploaded, include its path to update; otherwise keep existing DB value
    if ($newImagePath !== null) {
        $updateData['image'] = $newImagePath;
    }

    $slider->update($updateData);

    return redirect()->route('sliders.index')->with('success', 'تم تحديث شريط الصور بنجاح');
}

    public function destroy($id)
    {
        $slider = Slider::findOrFail($id);
        $slider->delete();
        return redirect()->route('sliders.index')->with('success', 'تم حذف الصورة بنجاح');
    }
}