<?php

namespace App\Http\Controllers\Investissement;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Log;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\TypeFormulaire;
use App\Models\Identite_personnel;
use App\Models\NatureProjet; 
use App\Models\SecteurFinancement;
use App\Models\Utilisateur;
use App\Models\Investissement;
use App\Models\ProjetDivertissementJeuneEnfant;
use App\Models\DirecteurEtablissement;
use App\Mail\JeuneEnfantStatutMail;
use Illuminate\Support\Facades\Mail;
use Carbon\Carbon;
use App\Notifications\JeuneEnfantStatutNotification;
use App\Exports\JeuneEnfantExport;
use Maatwebsite\Excel\Facades\Excel;
use Mpdf\Mpdf;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use Symfony\Component\HttpFoundation\StreamedResponse;
use App\Models\TpStatut;
use Illuminate\Support\Facades\Auth;
use App\Models\ImmobilierStatutHistory;

class JeuneEnfantController extends Controller
{
    public function index(Request $request)
{
    $query = ProjetDivertissementJeuneEnfant::with([
        'statut',
        'natureProjet',
        'investissements',
        'investissements.utilisateur'
    ]);

    if ($request->search_option == "nom") {

        $nom = trim($request->nom);

        if (
            $nom === "" ||                     // vide ou espace
            $nom === "%" ||                    // %
            $nom === "%%" ||                   // %%
            preg_match('/^[^a-zA-Zأ-ي0-9]+$/u', $nom) // caractères spéciaux uniquement
        ) {
            $query->whereRaw('1 = 0'); // renvoie zéro résultat
        } else {
            $query->whereHas('investissements.utilisateur', function ($q) use ($nom) {
                $q->where('nom_prenom', 'LIKE', "%$nom%");
            });
        }
    }

 
    if ($request->search_option == "nature" && $request->filled('nature')) {
        $query->where('id_nature', $request->nature);
    }

    if ($request->search_option == "identite" && $request->filled('type_identite')) {

        if ($request->type_identite == 1 && $request->filled('cin')) {
            $cin = trim($request->cin);

            if ($cin === "" || preg_match('/[^0-9]/', $cin)) {
                $query->whereRaw('1 = 0'); // invalide
            } else {
                $query->whereHas('investissements.utilisateur', fn($q) =>
                    $q->where('cin', $cin)
                );
            }
        }

        if ($request->type_identite == 2 && $request->filled('passport')) {
            $passport = trim($request->passport);

            if ($passport === "" || $passport === "%" || $passport === "%%") {
                $query->whereRaw('1 = 0');
            } else {
                $query->whereHas('investissements.utilisateur', fn($q) =>
                    $q->where('passport', $passport)
                );
            }
        }

        if ($request->type_identite == 3 && $request->filled('num_fiscal')) {
            $num = trim($request->num_fiscal);

            if ($num === "" || preg_match('/[^0-9]/', $num)) {
                $query->whereRaw('1 = 0');
            } else {
                $query->whereHas('investissements.utilisateur', fn($q) =>
                    $q->where('num_fiscal', $num)
                );
            }
        }
    }

    $projets = $query->orderByDesc('created_at')
                     ->paginate(10)
                     ->withQueryString();

    $natures = NatureProjet::all();

    return view('backoffice.investissement.gestion_jeune_enfant.index', compact('projets', 'natures'));
}


public function exportPdf(Request $request)
{
    $query = ProjetDivertissementJeuneEnfant::with([
        'statut',
        'natureProjet',
        'investissements.utilisateur'
    ]);

    // نفس فلاتر index
    if ($request->search_option === 'nom' && $request->filled('nom')) {
        $query->whereHas('investissements.utilisateur', function ($q) use ($request) {
            $q->where('nom_prenom', 'LIKE', '%' . trim($request->nom) . '%');
        });
    }

    if ($request->search_option === 'nature' && $request->filled('nature')) {
        $query->where('id_nature', $request->nature);
    }

    $projets = $query->latest()->get();

    $html = view(
        'backoffice.investissement.gestion_jeune_enfant.exports.jeune_enfant_pdf',
        compact('projets')
    )->render();

    $mpdf = new Mpdf([
        'mode' => 'utf-8',
        'format' => 'A4',
        'default_font' => 'dejavusans',
        'directionality' => 'rtl',
    ]);

    $mpdf->WriteHTML($html);

    return response($mpdf->Output('projets_jeunes_enfants.pdf', 'S'))
        ->header('Content-Type', 'application/pdf');
}



public function pdf($id)
{
    // =======================
    // 1️⃣ Charger le projet avec relations
    // =======================
    $projet = ProjetDivertissementJeuneEnfant::with([
        'directeur',
        'investissements',
        'investissements.utilisateur',
        'statut',
        'natureProjet'
    ])->findOrFail($id);

    $secteurs   = SecteurFinancement::all();
    $formulaire = TypeFormulaire::find($projet->id_formulaire);

    // =======================
    // 2️⃣ Forcer AR + RTL
    // =======================
    $lang = 'ar';
    $direction = 'rtl';

    // =======================
    // 3️⃣ Générer le HTML du PDF
    // =======================
    $html = view(
        'backoffice.investissement.gestion_jeune_enfant.pdf',
        compact(
            'projet',
            'secteurs',
            'formulaire',
            'direction',
            'lang'
        )
    )->render();

    // =======================
    // 4️⃣ Initialisation mPDF (AR)
    // =======================
    $mpdf = new Mpdf([
        'mode'              => 'utf-8',
        'format'            => 'A4',
        'default_font'      => 'dejavusans', // ممتاز للعربية
        'directionality'    => 'rtl',
        'autoScriptToLang'  => true,
        'autoLangToFont'    => true,
        'margin_top'        => 15,
        'margin_bottom'     => 15,
        'margin_left'       => 12,
        'margin_right'      => 12,
    ]);

    // Footer en arabe
    $mpdf->SetFooter('{PAGENO} / {nbpg}');

    // Écriture HTML
    $mpdf->WriteHTML($html);

    // =======================
    // 5️⃣ Affichage navigateur
    // =======================
    return $mpdf->Output(
        'jeune_enfant_'.$id.'.pdf',
        'I' // I = inline (navigateur)
    );
}


public function exportExcel(Request $request)
{
    // 🔹 نفس query متاع index
    $query = ProjetDivertissementJeuneEnfant::with([
        'statut',
        'natureProjet',
        'investissements',
        'investissements.utilisateur'
    ]);

    // 🔍 Filtre par nom
    if ($request->search_option == "nom") {
        $nom = trim($request->nom);

        if ($nom === "" || $nom === "%" || $nom === "%%" || preg_match('/^[^a-zA-Zأ-ي0-9]+$/u', $nom)) {
            $query->whereRaw('1 = 0');
        } else {
            $query->whereHas('investissements.utilisateur', function ($q) use ($nom) {
                $q->where('nom_prenom', 'LIKE', "%$nom%");
            });
        }
    }

    // 🔍 Filtre nature
    if ($request->search_option == "nature" && $request->filled('nature')) {
        $query->where('id_nature', $request->nature);
    }

    // 🔍 Filtre identité
    if ($request->search_option == "identite" && $request->filled('type_identite')) {

        if ($request->type_identite == 1 && $request->filled('cin')) {
            $query->whereHas('investissements.utilisateur', fn($q) =>
                $q->where('cin', trim($request->cin))
            );
        }

        if ($request->type_identite == 2 && $request->filled('passport')) {
            $query->whereHas('investissements.utilisateur', fn($q) =>
                $q->where('passport', trim($request->passport))
            );
        }

        if ($request->type_identite == 3 && $request->filled('num_fiscal')) {
            $query->whereHas('investissements.utilisateur', fn($q) =>
                $q->where('num_fiscal', trim($request->num_fiscal))
            );
        }
    }

    // ❌ لا paginate
    $projets = $query->orderByDesc('created_at')->get();

    // =========================
    // 📊 Excel
    // =========================
    $spreadsheet = new Spreadsheet();
    $sheet = $spreadsheet->getActiveSheet();
    $sheet->setRightToLeft(true);

    // 🟦 Header
    $sheet->fromArray([
        'المعرف',
        'الاسم و اللقب',
        'طبيعة المشروع',
        'وضعية الطلب'
    ], null, 'A1');

    // 🟩 Data
    $row = 2;
    foreach ($projets as $projet) {
        $sheet->setCellValue("A$row", $projet->id_projet);
        $sheet->setCellValue("B$row", $projet->investissements->first()?->utilisateur?->nom_prenom ?? '---');
        $sheet->setCellValue("C$row", $projet->natureProjet?->nature_ar ?? '---');
        $sheet->setCellValue("D$row", $projet->statut?->statut_ar ?? 'في الانتظار');
        $row++;
    }

    // 📥 Download
    $writer = new Xlsx($spreadsheet);
    $fileName = 'jeune_enfant_' . now()->format('Ymd_His') . '.xlsx';

    return response()->streamDownload(function () use ($writer) {
        $writer->save('php://output');
    }, $fileName, [
        'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
    ]);
}




    public function create()
    {
        $formulaire = TypeFormulaire::find(3);
        $typesIdentite = Identite_personnel::all();
        $naturesProjet = NatureProjet::all();
        $secteurs = SecteurFinancement::all();

        return view('backoffice.investissement.gestion_jeune_enfant.create', compact(
            'formulaire','typesIdentite','naturesProjet','secteurs'
        ));
    }

    public function store(Request $request)
    {
        DB::beginTransaction();
        try {
            $request->validate([
                // Utilisateur
                'nom_prenom'       => 'required|string|max:255',
                'id_type_identite' => 'required|integer|exists:tp_type_identite_pers,id_type',
                'cin'              => 'nullable|string|max:20',
                'date_delivrance_cin' => 'nullable|date',
                'lieu_delivrance_cin' => 'nullable|string|max:255',
                'passport'         => 'nullable|string|max:50',
                'num_fiscal'       => 'nullable|string|max:50',
                'tel'              => 'nullable|string|max:20',
                'adresse'          => 'nullable|string|max:255',
                'lieu_naissance'   => 'nullable|string|max:100',
                'date_naissance'   => 'nullable|date',
                'niveau_etude'     => 'nullable|string|max:100',
                'dernier_diplome'  => 'nullable|string|max:100',
                'experience'       => 'nullable|string|max:100',                
                // Directeur
                'nom_prenom_d'     => 'required|string|max:255',
                'num_cin'          => 'required|string|max:20',
                'date_delivrance'  => 'required|date',
                'lieu_delivrance'  => 'required|string|max:255',
                'niveau_etude'     => 'nullable|string|max:255',
                'dernier_diplome_obtenu' => 'nullable|string|max:255',

                // Investissement
                'investissements.*.secteur_invest_id' => 'required|integer|exists:tp_secteur_financement,id_secteur',
                'investissements.*.montant_inv'           => 'required|numeric',
                'investissements.*.source'            => 'nullable|string|max:255',
                'investissements.*.remarques'         => 'nullable|string',

                // Projet
                'nature_projet_id'          => 'required|integer|exists:tp_nature_projet,id_nature_p',
                'nbr_diplome_universitaire' => 'nullable|integer',
                'nbr_autre'                 => 'nullable|integer',
            ]);
            $utilisateur = Utilisateur::create([
                'nom_prenom'          => $request->nom_prenom,
                'id_type_identite'    => $request->id_type_identite,
                'cin'                 => $request->cin,
                'date_delivrance_cin' => $request->date_delivrance_cin,
                'lieu_delivrance_cin' => $request->lieu_delivrance_cin,
                'passport'            => $request->passport,
                'num_fiscal'          => $request->num_fiscal,
                'adresse'             => $request->adresse,
                'tel'                 => $request->tel,
                'lieu_naissance'      => $request->lieu_naissance,
                'date_naissance'      => $request->date_naissance,
                'niveau_etude'        => $request->niveau_etude,
                'dernier_diplome'     => $request->dernier_diplome,
                'experience'          => $request->experience,
            ]);
            $directeur = DirecteurEtablissement::create([
                'nom_prenom_d'           => $request->nom_prenom_d,
                'num_cin'                => $request->num_cin,
                'date_delivrance'        => $request->date_delivrance,
                'lieu_delivrance'        => $request->lieu_delivrance,
                'niveau_etude'           => $request->niveau_etude,
                'dernier_diplome_obtenu' => $request->dernier_diplome_obtenu,
            ]);
            $projet = ProjetDivertissementJeuneEnfant::create([
                'nature_projet_id'          => $request->nature_projet_id,
                'id_directeur'              => $directeur->id_directeur,
                'nbr_diplome_universitaire' => $request->nbr_diplome_universitaire,
                'nbr_autre'                 => $request->nbr_autre,
                'espace_spectacle'          => $request->espace_spectacle ?? 0,
                'caracteristique_spec'      => $request->caracteristique_spec,
                'espace_sportif'            => $request->espace_sportif ?? 0,
                'caracteristique_sportif'   => $request->caracteristique_sportif,
                'espace_activite_service'   => $request->espace_activite_service ?? 0,
                'caracteristique_activite_serv' => $request->caracteristique_activite_serv,
                'espace_activite_culturelle'=> $request->espace_activite_culturelle ?? 0,
                'caracteristique_activite_cult' => $request->caracteristique_activite_cult,
                'remarques'                 => $request->remarques,
                'id_statut'                 => 2,
                'id_formulaire'             => 3,
                'commentaire_motif'         => $request->commentaire_motif ?? 0,
            ]);
            if ($request->has('investissements')) {
                foreach ($request->investissements as $inv) {
                    Investissement::create([
                        'user_id'          => $utilisateur->id,
                        'secteur_invest_id'=> $inv['secteur_invest_id'],
                        'montant_inv'          => $inv['montant_inv'],
                        'source'           => $inv['source'] ?? null,
                        'remarques'        => $inv['remarques'] ?? null,
                        'id_projet'        => $projet->id_projet, 
                    ]);
                }
            }
            DB::commit();
            return redirect()->route('admin.jeune_enfant.index')
                            ->with('success', 'Projet enregistré avec succès ✅');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withErrors(['error' => 'Erreur lors de l\'enregistrement : '.$e->getMessage()])
                        ->withInput();
        }
    }
	
	
	
	public function toggleStatut(Request $request, $id)
{
    $request->validate([
        'id_statut' => 'required|integer',
        'commentaire_motif' => 'nullable|string',
        'commentaire_refus' => 'nullable|string',
        'commentaire_status' => 'nullable|string',
        'commentaire_acceptation' => 'nullable|string',
        'commentaire_avis' => 'nullable|string',
    ]);

    DB::beginTransaction();
    try {
        $projet = ProjetDivertissementJeuneEnfant::with('investissements.utilisateur')->findOrFail($id);

        $oldStatut = (int) ($projet->id_statut ?? 0);
        $newStatut = (int) $request->input('id_statut');

        // no-op guard
        if ($oldStatut === $newStatut) {
            DB::rollBack();
            return redirect()->back()->with('info', 'لا تغيير في الوضعية.');
        }

        // build comment (same logic as update)
        $comment = null;
        if ($request->filled('commentaire_refus')) {
            $comment = trim($request->input('commentaire_refus'));
        } elseif ($request->filled('commentaire_status')) {
            $comment = trim($request->input('commentaire_status'));
        } elseif ($request->filled('commentaire_motif')) {
            $comment = trim($request->input('commentaire_motif'));
        }
        if ($comment === '') $comment = null;

        // save comment into model if refusal (keep consistent with update)
        if ($newStatut == 3 && $request->filled('commentaire_motif')) {
            $projet->commentaire_motif = $request->input('commentaire_motif');
        } elseif ($comment !== null && $projet->commentaire_motif === null) {
            // optionally store generic comment if none exists
            $projet->commentaire_motif = $comment;
        }

        // persist new statut
        $projet->id_statut = $newStatut;
        $projet->save();

        // create history row (use null for FK when statut is 0 to avoid FK constraint issues)
        ImmobilierStatutHistory::create([
            'projet_id'      => $projet->id_projet,
            'from_statut_id' => $oldStatut ?: null,
            'to_statut_id'   => $newStatut ?: null,
            'changed_by'     => Auth::id() ?? null,
            'commentaire'    => $comment,
        ]);

        DB::commit();
    } catch (\Exception $e) {
        DB::rollBack();
        \Log::error("toggleStatut error for ProjetDivertissementJeuneEnfant ID {$id}: " . $e->getMessage(), [
            'trace' => $e->getTraceAsString(),
            'request' => $request->all(),
        ]);
        return back()->withErrors(['error' => 'حدث خطأ أثناء تغيير الحالة: ' . $e->getMessage()]);
    }

    // -------------------------
    // Mail + translation-aware DB notification (best-effort)
    // -------------------------

try {
    $user = $projet->investissements->first()?->utilisateur ?? null;

    // normalize numeric -> translation type
    $statusMap = [
        1 => 'accepte',
        3 => 'refuse',
        4 => 'incomplete',
        5 => 'deferred',
    ];
    $normType = $statusMap[$newStatut] ?? 'status_changed';

    // build translation params
    $translationParams = [];
    if ($normType === 'refuse') {
        $translationParams['reason'] = $request->input('commentaire_motif')
            ?? $request->input('commentaire_refus')
            ?? ($comment ?? '');
    } elseif ($normType === 'incomplete') {
        $translationParams['missing'] = $request->input('commentaire_acceptation') ?? ($comment ?? '');
    } elseif ($normType === 'deferred') {
        $translationParams['note'] = $request->input('commentaire_avis') ?? ($comment ?? '');
    }

    if ($user && !empty($user->email)) {
        try {
            // Always attempt to send a mail for the new status (best-effort)
            Mail::to($user->email)->send(
                new \App\Mail\JeuneEnfantStatutMail($normType, $user, $comment)
            );
        } catch (\Exception $mailEx) {
            \Log::error("Mail sending failed for projet {$projet->id_projet}: " . $mailEx->getMessage());
        }

        // DB notification (stores translation key + params)
        try {
            $user->notify(new \App\Notifications\JeuneEnfantStatutNotification(
                $normType,
                $translationParams,
                $projet->id_projet
            ));
        } catch (\Exception $notifEx) {
            \Log::error("Notification saving failed for projet {$projet->id_projet}: " . $notifEx->getMessage());
        }
    } else {
        \Log::warning("Aucun utilisateur avec email trouvé pour le projet ID: {$projet->id_projet}");
    }
} catch (\Exception $ex) {
    \Log::error("Notification/mail error for ProjetDivertissementJeuneEnfant ID {$projet->id_projet}: " . $ex->getMessage());
    // do not rollback — status & history already saved
}


    return redirect()->back()->with('success', 'تم تغيير حالة الطلب بنجاح');
}

	
/*
public function toggleStatut(Request $request, $id)
{
    $projet = ProjetDivertissementJeuneEnfant::with('investissements.utilisateur')->findOrFail($id);
    $projet->id_statut = $request->id_statut;

    $user = $projet->investissements->first()?->utilisateur;

    // prepare comment
    $comment = null;
    if ($request->filled('commentaire_refus')) {
        $comment = trim($request->input('commentaire_refus'));
    } elseif ($request->filled('commentaire_status')) {
        $comment = trim($request->input('commentaire_status'));
    } elseif ($request->filled('commentaire_motif')) {
        $comment = trim($request->input('commentaire_motif'));
    }
    if ($comment === '') $comment = null;

    // save comment into model if refusal
    if ($request->id_statut == 3 && $request->filled('commentaire_motif')) {
        $projet->commentaire_motif = $request->commentaire_motif;
    }

    $projet->save();

    // build normalized type & params for translation
    $statusMap = [
        1 => 'accepted',
        3 => 'rejected',
        4 => 'incomplete',
        5 => 'deferred',
    ];
    $normType = $statusMap[$request->id_statut] ?? 'status_changed';
    $translationParams = [];

    if ($normType === 'rejected') {
        $translationParams['reason'] = $request->input('commentaire_motif') 
                                        ?? $request->input('commentaire_refus') 
                                        ?? ($comment ?? '');
    } elseif ($normType === 'incomplete') {
        $translationParams['missing'] = $request->input('commentaire_acceptation') ?? ($comment ?? '');
    } elseif ($normType === 'deferred') {
        $translationParams['note'] = $request->input('commentaire_avis') ?? ($comment ?? '');
    }

    // send mail + translated notification
    if ($user && !empty($user->email)) {
        try {
            if ($request->id_statut == 3 && $request->commentaire_motif) {
                Mail::to($user->email)->send(
                    new JeuneEnfantStatutMail('refuse', $user, null, $request->commentaire_motif)
                );
            } elseif ($request->id_statut == 1) {
                Mail::to($user->email)->send(
                    new JeuneEnfantStatutMail('accepte', $user)
                );
            }

            // DB notification carries translation key/params (via JeuneEnfantStatutNotification)
            $user->notify(new JeuneEnfantStatutNotification(
                $normType,
                $translationParams,
                $projet->id_projet
            ));
        } catch (\Exception $e) {
            \Log::error("Notification/mail error for ProjetDivertissementJeuneEnfant toggleStatut ID {$projet->id_projet}: " . $e->getMessage());
        }
    } else {
        Log::warning("Aucun utilisateur avec email trouvé pour le projet ID: {$id}");
    }

    return redirect()->back()->with('success', 'تم تغيير حالة الطلب بنجاح');
}

*/
    public function show($id)
    {
        $projet = ProjetDivertissementJeuneEnfant::with([
        'directeur',
        'investissements',
        'investissements.utilisateur',
        // eager load histories coming from ImmobilierStatutHistory
        'statutHistories.fromStatut',
        'statutHistories.toStatut',
        'statutHistories.changer'
    ])->findOrFail($id);

        $typesIdentite = Identite_personnel::all();
        $naturesProjet = NatureProjet::all();
        $secteurs = SecteurFinancement::all();
        $formulaire = TypeFormulaire::find($projet->id_formulaire);
         $statuts = TpStatut::orderBy('id_statut')->get();
        return view('backoffice.investissement.gestion_jeune_enfant.show', compact(
            'projet', 'typesIdentite', 'naturesProjet', 'secteurs', 'formulaire','statuts'
        ));
    }

    public function edit($id)
    {
         $projet = ProjetDivertissementJeuneEnfant::with([
        'directeur',
        'investissements',
        'investissements.utilisateur',
        // eager load histories coming from ImmobilierStatutHistory
        'statutHistories.fromStatut',
        'statutHistories.toStatut',
        'statutHistories.changer'
    ])->findOrFail($id);
        $typesIdentite = Identite_personnel::all();
        $naturesProjet = NatureProjet::all();
        $secteurs = SecteurFinancement::all();
        $formulaire = TypeFormulaire::find($projet->id_formulaire);
        $statuts = TpStatut::orderBy('id_statut')->get();
        return view('backoffice.investissement.gestion_jeune_enfant.edit', compact(
            'projet', 'typesIdentite', 'naturesProjet', 'secteurs', 'formulaire', 'statuts'
        ));
    }

public function update(Request $request, $id)
{
    DB::beginTransaction();

    try {
        $projet = ProjetDivertissementJeuneEnfant::findOrFail($id);

        
        $request->validate([
            // Utilisateur
            'nom_prenom'       => 'required|string|max:255',
            'id_type_identite' => 'required|integer|exists:tp_type_identite_pers,id_type',
            'cin'              => 'nullable|string|max:20',
            'date_delivrance_cin' => 'nullable|date',
            'lieu_delivrance_cin' => 'nullable|string|max:255',
            'passport'         => 'nullable|string|max:50',
            'num_fiscal'       => 'nullable|string|max:50',
            'tel'              => 'nullable|string|max:20',
            'adresse'          => 'nullable|string|max:255',
            'lieu_naissance'   => 'nullable|string|max:100',
            'date_naissance'   => 'nullable|date',
            'niveau_etude_user'     => 'nullable|string|max:100',
				'dernier_diplome_user'  => 'nullable|string|max:100',
				'experience_user'       => 'nullable|string|max:150',

            // Directeur
            'nom_prenom_d'     => 'required|string|max:255',
            'num_cin'          => 'required|string|max:20',
            'date_delivrance'  => 'required|date',
            'lieu_delivrance'  => 'required|string|max:255',
            'niveau_etude'     => 'nullable|string|max:255',
            'dernier_diplome_obtenu' => 'nullable|string|max:255',

            // Investissement
            'investissements.*.secteur_invest_id' => 'required|integer|exists:tp_secteur_financement,id_secteur',
            'investissements.*.montant_inv'       => 'required|numeric',
            'investissements.*.source'            => 'nullable|string|max:255',
            'investissements.*.remarques'         => 'nullable|string',

            // Projet
            'nature_projet_id'          => 'required|integer|exists:tp_nature_projet,id_nature_p',
            'nbr_diplome_universitaire' => 'nullable|integer',
            'nbr_autre'                 => 'nullable|integer',
            'id_statut'                 => 'nullable|in:1,3,4,5,2,6', // adapt as needed
        ]);


			/*
			$rules = [
						 'nom_prenom'       => 'required|string|max:255',
						'id_type_identite' => 'required|integer|exists:tp_type_identite_pers,id_type',
						'date_delivrance_cin' => 'nullable|date',
						'lieu_delivrance_cin' => 'nullable|string|max:255',
						'tel'              => 'nullable|string|max:20',
						'adresse'          => 'nullable|string|max:255',
						'lieu_naissance'   => 'nullable|string|max:100',
						'date_naissance'   => 'nullable|date',
						'niveau_etude'     => 'nullable|string|max:100',
						'dernier_diplome'  => 'nullable|string|max:100',
						'experience'       => 'nullable|string|max:100',
			'cin' => 'nullable|digits:8|unique:utilisateur,cin,' . $projet->utilisateur->id . ',id',
				'passport' => 'nullable|digits_between:8,12|unique:utilisateur,passport,' . $projet->utilisateur->id . ',id',
				'num_fiscal' => 'nullable|string|unique:utilisateur,num_fiscal,' . $projet->utilisateur->id . ',id',

						// Directeur
						'nom_prenom_d'     => 'required|string|max:255',
						'num_cin'          => 'required|string|max:20',
						'date_delivrance'  => 'required|date',
						'lieu_delivrance'  => 'required|string|max:255',
						'niveau_etude'     => 'nullable|string|max:255',
						'dernier_diplome_obtenu' => 'nullable|string|max:255',

						// Investissement
						'investissements.*.secteur_invest_id' => 'required|integer|exists:tp_secteur_financement,id_secteur',
						'investissements.*.montant_inv'       => 'required|numeric',
						'investissements.*.source'            => 'nullable|string|max:255',
						'investissements.*.remarques'         => 'nullable|string',

						// Projet
						'nature_projet_id'          => 'required|integer|exists:tp_nature_projet,id_nature_p',
						'nbr_diplome_universitaire' => 'nullable|integer',
						'nbr_autre'                 => 'nullable|integer',
						'id_statut'                 => 'nullable|in:1,3,4,5,2,6', // adapt as needed
						];
			$type = (int) $request->id_type_identite;

			// CIN
			if ($type === 1) {
				$rules['cin'] = 'required|digits:8|unique:utilisateur,cin,' . $immobilier->utilisateur->id . ',id';
			}

			// PASSPORT
			if ($type === 2) {
				$rules['passport'] = 'required|digits_between:8,12|unique:utilisateur,passport,' . $immobilier->utilisateur->id . ',id';
			}

			// NUM FISCAL
			if ($type === 3) {
				$rules['num_fiscal'] = [
					'required',
					'string',
					'unique:utilisateur,num_fiscal,' . $immobilier->utilisateur->id . ',id',
					'regex:/^[A-Za-z0-9]{7}\/[A-Za-z]\/[A-Za-z]\/[A-Za-z]\/[A-Za-z0-9]{3}$/'
				];
			}
			$request->validate($rules);

			*/

        // Keep old/new statut for history
        $oldStatut = (int) ($projet->id_statut ?? 0);
        $newStatut = (int) ($request->input('id_statut') ?? $oldStatut);

        // Update associated utilisateur (if any) - you used first investment's user earlier
        $utilisateur = $projet->investissements->first()?->utilisateur;
        if ($utilisateur) {
            $utilisateur->update([
                'nom_prenom'          => $request->nom_prenom,
                'id_type_identite'    => $request->id_type_identite,
                'cin'                 => $request->cin,
                'date_delivrance_cin' => $request->date_delivrance_cin,
                'lieu_delivrance_cin' => $request->lieu_delivrance_cin,
                'passport'            => $request->passport,
                'num_fiscal'          => $request->num_fiscal,
                'adresse'             => $request->adresse,
                'tel'                 => $request->tel,
                'lieu_naissance'      => $request->lieu_naissance,
                'date_naissance'      => $request->date_naissance,
                
            ]);
        }

        // Update directeur
        $directeur = $projet->directeur;
        if ($directeur) {
            $directeur->update([
                'nom_prenom_d'           => $request->nom_prenom_d,
                'num_cin'                => $request->num_cin,
                'date_delivrance'        => $request->date_delivrance,
                'lieu_delivrance'        => $request->lieu_delivrance,
                'niveau_etude'           => $request->niveau_etude,
                'dernier_diplome_obtenu' => $request->dernier_diplome_obtenu,
            ]);
        }

        // Prepare comment selection (prefer dedicated fields, then fallback to commentaire_motif)
        $comment = null;
        if ($request->filled('commentaire_refus')) {
            $comment = trim($request->input('commentaire_refus'));
        } elseif ($request->filled('commentaire_status')) {
            $comment = trim($request->input('commentaire_status'));
        } elseif ($request->filled('commentaire_motif')) {
            $comment = trim($request->input('commentaire_motif'));
        }
        if ($comment === '') $comment = null;

        // Update projet
        $projet->update([
            'nature_projet_id'          => $request->nature_projet_id,
            'id_statut'                 => $newStatut,
            'nbr_diplome_universitaire' => $request->nbr_diplome_universitaire,
            'nbr_autre'                 => $request->nbr_autre,
            'espace_spectacle'          => $request->has('espace_spectacle') ? ($request->espace_spectacle ? 1 : 0) : ($projet->espace_spectacle ?? 0),
            'caracteristique_spec'      => $request->caracteristique_spec,
            'espace_sportif'            => $request->has('espace_sportif') ? ($request->espace_sportif ? 1 : 0) : ($projet->espace_sportif ?? 0),
            'caracteristique_sportif'   => $request->caracteristique_sportif,
            'espace_activite_service'   => $request->has('espace_activite_service') ? ($request->espace_activite_service ? 1 : 0) : ($projet->espace_activite_service ?? 0),
            'caracteristique_activite_serv' => $request->caracteristique_activite_serv,
            'espace_activite_culturelle'=> $request->has('espace_activite_culturelle') ? ($request->espace_activite_culturelle ? 1 : 0) : ($projet->espace_activite_culturelle ?? 0),
            'caracteristique_activite_cult' => $request->caracteristique_activite_cult,
            'remarques'                 => $request->remarques,
            'id_formulaire'             => $projet->id_formulaire ?? 3,
            'commentaire_motif'         => $comment, // store selected comment on the projet
			'niveau_etude_user'  => $request->niveau_etude_user,
				'dernier_diplome_user' => $request->dernier_diplome_user,
				'experience_user'    => $request->experience_user,
				'id_user'            => $utilisateur->id, 
        ]);

        // Replace investissements if provided
        if ($request->has('investissements')) {
            // If there's no utilisateur, we still create items but set user_id null (or handle as you prefer)
            $projet->investissements()->delete();
            foreach ($request->investissements as $inv) {
                Investissement::create([
                    'user_id'           => $utilisateur?->id ?? null,
                    'secteur_invest_id' => $inv['secteur_invest_id'],
                    'montant_inv'       => $inv['montant_inv'],
                    'source'            => $inv['source'] ?? null,
                    'remarques'         => $inv['remarques'] ?? null,
                    'id_projet'         => $projet->id_projet,
                ]);
            }
        }

        // If statut changed, create a history row linked to projet_id
        if ($oldStatut !== $newStatut) {
            ImmobilierStatutHistory::create([
                // link to projet table using correct primary key
                'projet_id'     => $projet->id_projet,
                'from_statut_id'=> $oldStatut ?: null,
                'to_statut_id'  => $newStatut ?: null,
                'changed_by'    => Auth::id() ?? null,
                'commentaire'   => $comment,
            ]);
        }

        DB::commit();

// après DB::commit(); -> envoi mails/notifications
try {
    $user = $utilisateur ?? null;

    // normalization map
    $statusMap = [
        1 => 'accepte',
        3 => 'refuse',
        4 => 'incomplete',
        5 => 'deferred',
    ];
    $normType = $statusMap[$newStatut] ?? 'status_changed';

    // build translation params
    $translationParams = [];
    if ($normType === 'refuse') {
        $translationParams['reason'] = $request->input('commentaire_refus')
            ?? $request->input('commentaire_motif')
            ?? ($comment ?? '');
    } elseif ($normType === 'incomplete') {
        $translationParams['missing'] = $request->input('commentaire_acceptation') ?? ($comment ?? '');
    } elseif ($normType === 'deferred') {
        $translationParams['note'] = $request->input('commentaire_avis') ?? ($comment ?? '');
    }

    if ($user && !empty($user->email)) {
        try {
            // send a mail for the normalized status (use $comment as motif where relevant)
            Mail::to($user->email)->send(
                new \App\Mail\JeuneEnfantStatutMail($normType, $user, $comment)
            );
        } catch (\Exception $mailEx) {
            \Log::error("Mail sending failed for projet {$projet->id_projet}: " . $mailEx->getMessage());
            // don't throw — notifications are best-effort
        }

        // send DB notification with translation key + params
        try {
            $user->notify(new \App\Notifications\JeuneEnfantStatutNotification(
                $normType,
                $translationParams,
                $projet->id_projet
            ));
        } catch (\Exception $notifEx) {
            \Log::error("Notification saving failed for projet {$projet->id_projet}: " . $notifEx->getMessage());
        }
    } else {
        \Log::warning("No user or missing email for projet ID: {$projet->id_projet}");
    }
} catch (\Exception $ex) {
    \Log::error("Notification/mail error for ProjetDivertissementJeuneEnfant ID {$projet->id_projet}: " . $ex->getMessage());
}


        
        return redirect()->route('admin.jeune_enfant.index')
                         ->with('success', 'تم تحديث المشروع بنجاح');
    } catch (\Exception $e) {
        DB::rollBack();

        // Optional: log the exception for debugging
        \Log::error('ProjetDivertissementJeuneEnfant update error: ' . $e->getMessage(), [
            'id' => $id,
            'trace' => $e->getTraceAsString()
        ]);

        return back()->withErrors(['error' => 'Erreur lors de la mise à jour : ' . $e->getMessage()])
                     ->withInput();
    }
}

    public function destroy($id)
    {
        DB::beginTransaction();
        try {
            $projet = ProjetDivertissementJeuneEnfant::with(['investissements', 'directeur', 'investissements.utilisateur'])->findOrFail($id);
            foreach ($projet->investissements as $investissement) {
                $investissement->delete();
            }
            if ($projet->directeur) {
                $projet->directeur->delete();
            }
            $projet->delete();
            DB::commit();
            return redirect()->route('admin.jeune_enfant.index')
                            ->with('success', 'تم حذف المشروع بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->route('admin.jeune_enfant.index')
                            ->withErrors(['error' => 'Erreur lors de la suppression : ' . $e->getMessage()]);
        }
    }


}
