<?php
namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\FieldDefinition;
use Illuminate\Validation\Rule;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
class FieldDefinitionController extends Controller
{
    public function index()
    {
        $definitions = FieldDefinition::orderBy('sort_order')->get();
        return view('backoffice.concours_ext.fields_index', compact('definitions'));
    }

    public function create()
    {
        // blank model for the form
        $fieldDefinition = new FieldDefinition();
        return view('backoffice.concours_ext.fields', compact('fieldDefinition'));
    }


protected function generateUniqueKeyFromLabel(string $label): string
{
    // base key: latin letters, numbers, underscores allowed
    $base = Str::slug(mb_strtolower($label), '_'); // produces letters_numbers_like
    if ($base === '') {
        $base = 'field';
    }
    $key = $base;
    $i = 1;
    while (\App\Models\FieldDefinition::where('key', $key)->exists()) {
        $key = $base . '_' . $i++;
    }
    return $key;
}

public function store(Request $request)
{
    $data = $request->validate([
        'label' => ['required', 'string', 'max:255'],
        'input_type' => ['required', 'string', 'max:30'],
        'options_json' => ['nullable', 'string'],
        'is_visible' => ['nullable', 'boolean'],
        'is_required' => ['nullable', 'boolean'],
        'sort_order' => ['nullable', 'integer'],
    ]);

    $data['is_visible'] = $request->has('is_visible') ? 1 : 0;
    $data['is_required'] = $request->has('is_required') ? 1 : 0;

    // Normalize options: allow entering CSV / newline list
    if (!empty($data['options_json'])) {
        json_decode($data['options_json']);
        if (json_last_error() !== JSON_ERROR_NONE) {
            $opts = array_values(array_filter(array_map('trim', preg_split("/[\r\n,]+/", $data['options_json']))));
            $data['options_json'] = json_encode($opts, JSON_UNESCAPED_UNICODE);
        } else {
            // ensure JSON is array if user provided JSON
            $v = json_decode($data['options_json'], true);
            if (!is_array($v)) {
                $opts = array_values(array_filter(array_map('trim', preg_split("/[\r\n,]+/", $data['options_json']))));
                $data['options_json'] = json_encode($opts, JSON_UNESCAPED_UNICODE);
            } else {
                $data['options_json'] = json_encode(array_values($v), JSON_UNESCAPED_UNICODE);
            }
        }
    }

    // default sort_order if empty: push to end
    if (empty($data['sort_order'])) {
        $max = FieldDefinition::max('sort_order');
        $data['sort_order'] = $max ? ($max + 1) : 1;
    }

    // auto-generate a unique key from the label (no key input required)
    $data['key'] = $this->generateUniqueKeyFromLabel($data['label']);

    // keep value_type null by default (you removed it from UI)
    $data['value_type'] = $request->input('value_type', null);

    FieldDefinition::create($data);

    return redirect()->route('backoffice.field_definitions.index')->with('success', 'تم إضافة الحقل بنجاح.');
}

public function update(Request $request, FieldDefinition $fieldDefinition)
{
    $data = $request->validate([
        'label' => ['required', 'string', 'max:255'],
        'input_type' => ['required', 'string', 'max:30'],
        'options_json' => ['nullable', 'string'],
        'is_visible' => ['nullable', 'boolean'],
        'is_required' => ['nullable', 'boolean'],
        'sort_order' => ['nullable', 'integer'],
    ]);

    $data['is_visible'] = $request->has('is_visible') ? 1 : 0;
    $data['is_required'] = $request->has('is_required') ? 1 : 0;

    if (!empty($data['options_json'])) {
        json_decode($data['options_json']);
        if (json_last_error() !== JSON_ERROR_NONE) {
            $opts = array_values(array_filter(array_map('trim', preg_split("/[\r\n,]+/", $data['options_json']))));
            $data['options_json'] = json_encode($opts, JSON_UNESCAPED_UNICODE);
        } else {
            $v = json_decode($data['options_json'], true);
            if (!is_array($v)) {
                $opts = array_values(array_filter(array_map('trim', preg_split("/[\r\n,]+/", $data['options_json']))));
                $data['options_json'] = json_encode($opts, JSON_UNESCAPED_UNICODE);
            } else {
                $data['options_json'] = json_encode(array_values($v), JSON_UNESCAPED_UNICODE);
            }
        }
    }

    // do not change the 'key' on update (we removed key from UI)
    // preserve any existing value_type (we don't touch it)
    $fieldDefinition->update($data);

    return redirect()->route('backoffice.field_definitions.index')->with('success', 'تم تحديث الحقل بنجاح.');
}

    public function edit(FieldDefinition $fieldDefinition)
    {
        return view('backoffice.concours_ext.edit_field', compact('fieldDefinition'));
    }

public function destroy(Request $request, $id)
{
    DB::beginTransaction();
    try {
        // Try to fetch the model — do NOT use implicit binding here so we can handle "not found"
        $field = FieldDefinition::find($id);

        // If not found, return a friendly JSON (AJAX) or redirect (normal form)
        if (!$field) {
            DB::rollBack();
            if ($request->wantsJson() || $request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'الحقل غير موجود أو تم حذفه سابقاً.'
                ], 404);
            }
            return redirect()->route('backoffice.field_definitions.index')
                             ->withErrors(['general' => 'الحقل غير موجود أو تم حذفه سابقاً.']);
        }

        // 1) detach from pivot (concours_field)
        DB::table('concours_field')->where('field_definition_id', $field->id)->delete();

        // 2) delete any files on disk and DB file records
        $files = $field->files()->get();
        foreach ($files as $f) {
            if (!empty($f->file_path) && Storage::exists($f->file_path)) {
                try {
                    Storage::delete($f->file_path);
                } catch (\Throwable $ex) {
                    Log::warning("Failed to delete file for FieldFile {$f->id}: ".$ex->getMessage());
                }
            }
            $f->delete();
        }

        // 3) delete stored values for this field
        $field->values()->delete();

        // 4) delete the field itself
        $field->delete();

        DB::commit();

        if ($request->wantsJson() || $request->ajax()) {
            return response()->json(['success' => true, 'message' => 'تم حذف الحقل.']);
        }
        return redirect()->route('backoffice.field_definitions.index')->with('success', 'تم حذف الحقل بنجاح.');
    } catch (\Throwable $e) {
        DB::rollBack();
        Log::error('FieldDefinition destroy error: '.$e->getMessage(), [
            'id' => $id, 'trace' => $e->getTraceAsString()
        ]);
        if ($request->wantsJson() || $request->ajax()) {
            return response()->json(['success' => false, 'message' => 'حدث خطأ أثناء الحذف.'], 500);
        }
        return back()->withErrors(['general' => 'حدث خطأ أثناء الحذف.']);
    }
}

  public function toggleVisibility(FieldDefinition $fieldDefinition)
    {
        $fieldDefinition->is_visible = !$fieldDefinition->is_visible;
        $fieldDefinition->save();
        return redirect()->route('backoffice.field_definitions.index')->with('success', 'تم تغيير حالة الظهور.');
    }

    public function reorder(Request $request)
    {
        // expects input order: [id1, id2, id3]
        $order = $request->input('order', []);
        if (!is_array($order)) {
            return back()->with('error', 'طلب غير صالح.');
        }
        foreach ($order as $index => $id) {
            FieldDefinition::where('id', $id)->update(['sort_order' => $index + 1]);
        }
        return redirect()->route('backoffice.field_definitions.index')->with('success', 'تم حفظ ترتيب الحقول.');
    }
}
