<?php

namespace App\Http\Controllers;

use App\Models\Federation;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;

class FederationController extends Controller
{
    public function index(Request $request)
    {
        $search = trim($request->get('search'));
        $words = $search ? preg_split('/\s+/', $search) : [];

        $federations = Federation::when($words, function ($query) use ($words) {
            foreach ($words as $word) {
                $word = str_replace(['\\', '%', '_'], ['\\\\', '\\%', '\\_'], $word);
                $query->where(function ($q) use ($word) {
                    $q->where('nom_ar', 'LIKE', "%{$word}%")
                    ->orWhere('nom_fr', 'LIKE', "%{$word}%")
                    ->orWhere('nom_en', 'LIKE', "%{$word}%")
                    ->orWhere('president_ar', 'LIKE', "%{$word}%")
                    ->orWhere('president_fr', 'LIKE', "%{$word}%")
                    ->orWhere('president_en', 'LIKE', "%{$word}%");
                });
            }
        })
        ->orderBy('created_at','desc')
        ->paginate(10)
        ->appends(['search' => $search]);

        
        $computeLogoSrc = function ($stored) {
            if (empty($stored)) return null;

            if (filter_var($stored, FILTER_VALIDATE_URL)) {
                return $stored;
            }

            $candidates = [
                $stored,
                '/www/mjs/shared_uploads/images/' . basename($stored),
                public_path(ltrim($stored, '/')),
                storage_path('app/' . ltrim($stored, '/')),
            ];

            foreach ($candidates as $path) {
                if (@file_exists($path) && @is_file($path) && @is_readable($path)) {
                    $mime = @mime_content_type($path) ?: 'image/png';
                    $contents = @file_get_contents($path);
                    if ($contents !== false) {
                        return "data:{$mime};base64," . base64_encode($contents);
                    }
                }
            }

            if (strpos($stored, '/') === 0) {
                return url(ltrim($stored, '/'));
            }

            return null;
        };

        foreach ($federations as $f) {
            $f->logoSrc = $computeLogoSrc($f->logo);
        }

        return view('backoffice.federation.index', compact('federations'));
    }


    public function create()
    {
        return view('backoffice.federation.create');
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'nom_ar' => 'required|string|max:255',
            'nom_fr' => 'nullable|string|max:255',
            'nom_en' => 'nullable|string|max:255',
            'president_ar' => 'nullable|string|max:150',
            'president_fr' => 'nullable|string|max:150',
            'president_en' => 'nullable|string|max:150',
            'telephone' => 'nullable|string|max:50',
            'telephone2' => 'nullable|string|max:50',
            'fax' => 'nullable|string|max:50',
            'email' => 'nullable|string|max:200',
            'site_web' => 'nullable|url|max:255',
            'adresse_ar' => 'nullable|string',
            'adresse_fr' => 'nullable|string',
            'adresse_en' => 'nullable|string',
            'date_creation' => 'nullable|date',
            'logo' => 'nullable|image|mimes:png,jpg,jpeg,webp|max:2048',
        ]);

        $candidates = [
            '/www/mjs/shared_uploads/images',
            '/home/preprov/www/mjs/shared_uploads/images',
            base_path('shared_uploads/images'),
            base_path('../shared_uploads/images'),
            public_path('../../shared_uploads/images'),
        ];

        $usable = null;
        $tried = [];

        if ($request->hasFile('logo')) {
            $file = $request->file('logo');

            foreach ($candidates as $cand) {
                $rp = @realpath($cand);
                $tried[$cand] = [
                    'realpath' => $rp ?: null,
                    'exists' => @file_exists($cand) ? 'yes' : 'no',
                    'is_dir' => @is_dir($cand) ? 'yes' : 'no',
                    'is_writable' => @is_writable($cand) ? 'yes' : 'no'
                ];
                if ($rp && is_dir($rp) && is_writable($rp)) {
                    $usable = $rp;
                    break;
                }
            }

            if (!$usable) {
                $msg = "Upload folder not accessible by PHP. Checked: ";
                foreach ($tried as $c => $info) {
                    $msg .= "[{$c} -> realpath=" . ($info['realpath'] ?? 'null') .
                        ", exists={$info['exists']}, is_dir={$info['is_dir']}, writable={$info['is_writable']}] ";
                }
                return back()->withInput()->withErrors(['logo' => $msg]);
            }

            // sanitize filename
            $originalName = $file->getClientOriginalName();
            $originalName = basename($originalName);
            $originalName = preg_replace('/[\0\/\\\\<>:"|?*\x00-\x1F]+/u', '-', $originalName);
            if ($originalName === '' || $originalName === '.' || $originalName === '..') {
                return back()->withInput()->withErrors(['logo' => 'اسم الملف الأصلي غير صالح.']);
            }

            $targetPath = rtrim($usable, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $originalName;

            // prevent overwrite
            if (file_exists($targetPath)) {
                $dotPos = strrpos($originalName, '.');
                if ($dotPos !== false) {
                    $base = substr($originalName, 0, $dotPos);
                    $ext  = substr($originalName, $dotPos);
                } else {
                    $base = $originalName;
                    $ext = '';
                }
                $originalName = $base . '-' . time() . $ext;
                $targetPath = rtrim($usable, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $originalName;
            }

            try {
                $file->move($usable, $originalName);
                // save absolute path in DB
                $data['logo'] = $targetPath;
            } catch (\Throwable $e) {
                \Log::error('Federation logo move failed: ' . $e->getMessage(), ['exception' => $e]);
                return back()->withInput()->withErrors(['logo' => 'فشل حفظ الشعار على الخادم.']);
            }
        } else {
            $data['logo'] = null;
        }

        $data['user_id'] = auth()->id();

        \App\Models\Federation::create($data);

        return redirect()
            ->route('federation.index')
            ->with('success', 'تمت إضافة الجامعة بنجاح');
    }

    /*public function edit(Federation $federation)
    {
        return view('backoffice.federation.edit', compact('federation'));
    }*/

        public function edit(Federation $federation)
{
    $computeLogoSrc = function ($stored) {
        if (empty($stored)) return null;
        if (filter_var($stored, FILTER_VALIDATE_URL)) return $stored;

        $candidates = [
            $stored,
            '/www/mjs/shared_uploads/images/' . basename($stored),
            public_path(ltrim($stored, '/')),
            storage_path('app/' . ltrim($stored, '/')),
        ];

        foreach ($candidates as $path) {
            if (@file_exists($path) && @is_file($path) && @is_readable($path)) {
                $mime = @mime_content_type($path) ?: 'image/png';
                $contents = @file_get_contents($path);
                if ($contents !== false) {
                    return "data:{$mime};base64," . base64_encode($contents);
                }
            }
        }

        if (strpos($stored, '/') === 0) {
            return url(ltrim($stored, '/'));
        }

        return null;
    };

    $federation->logoSrc = $computeLogoSrc($federation->logo);

    return view('backoffice.federation.edit', compact('federation'));
}


    public function update(Request $request, Federation $federation)
    {
        $data = $request->validate([
            'nom_ar' => 'required|string|max:255',
            'nom_fr' => 'nullable|string|max:255',
            'nom_en' => 'nullable|string|max:255',
            'president_ar' => 'nullable|string|max:150',
            'president_fr' => 'nullable|string|max:150',
            'president_en' => 'nullable|string|max:150',
            'telephone' => 'nullable|string|max:50',
            'email' => 'nullable|string|max:200',
            'telephone2' => 'nullable|string|max:50',
            'fax' => 'nullable|string|max:50',
            'site_web' => 'nullable|url|max:255',
            'adresse_ar' => 'nullable|string',
            'adresse_fr' => 'nullable|string',
            'adresse_en' => 'nullable|string',
            'date_creation' => 'nullable|date',
            'logo' => 'nullable|image|mimes:png,jpg,jpeg,webp|max:2048',
        ]);

        $candidates = [
            '/www/mjs/shared_uploads/images',
            '/home/preprov/www/mjs/shared_uploads/images',
            base_path('shared_uploads/images'),
            base_path('../shared_uploads/images'),
            public_path('../../shared_uploads/images'),
        ];

        $usable = null;

        // تحديث الشعار إذا تم رفعه
        if ($request->hasFile('logo')) {
            $file = $request->file('logo');

            foreach ($candidates as $cand) {
                $rp = @realpath($cand);
                if ($rp && is_dir($rp) && is_writable($rp)) {
                    $usable = $rp;
                    break;
                }
            }

            if (!$usable) {
                return back()->withInput()->withErrors(['logo' => 'مجلد رفع الشعار غير قابل للكتابة.']);
            }

            $originalName = basename($file->getClientOriginalName());
            $originalName = preg_replace('/[\0\/\\\\<>:"|?*\x00-\x1F]+/u', '-', $originalName);
            $targetPath = rtrim($usable, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $originalName;

            if (file_exists($targetPath)) {
                $dotPos = strrpos($originalName, '.');
                $base = $dotPos !== false ? substr($originalName, 0, $dotPos) : $originalName;
                $ext = $dotPos !== false ? substr($originalName, $dotPos) : '';
                $originalName = $base . '-' . time() . $ext;
                $targetPath = rtrim($usable, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $originalName;
            }

            $file->move($usable, $originalName);
            $data['logo'] = $targetPath;
        }

        $federation->update($data);

        return redirect()
            ->route('federation.index')
            ->with('success', 'تم تعديل بيانات الجامعة بنجاح.');
    }

    public function destroy(Federation $federation)
    {
        if ($federation->logo && file_exists($federation->logo)) {
            @unlink($federation->logo);
        }

        $federation->delete();

        return redirect()
            ->route('federation.index')
            ->with('success', 'تم حذف الجامعة بنجاح.');
    }



}