<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\MilieuAuberge;
use App\Models\Auberge;
use App\Models\Gouvernorat;
use App\Models\TpSportsAuberge;
use App\Models\TpServiceAuberge;
use App\Models\Delegation;
use App\Models\RestaurationAuberge;
use App\Models\TypeChambre;
use App\Models\ChambreTente;
use App\Models\EspaceAuberge;
use App\Models\TerrainSportifAuberge;
use App\Models\ServiceAuberge;
use App\Models\TpTypeTarif;
use App\Models\TarifAuberge;

class AubergesController extends Controller
{
	public function index(Request $request)
	{
		$search = (string) $request->input('q', '');

		$search = trim($search);

		if ($search === '' || preg_match('/^[\s%]+$/', $search)) {
			$search = null;
		}

		$query = Auberge::with(['gouvernorat', 'delegation']);

		if ($search) {
			$words = preg_split('/\s+/', $search);

			$query->where(function ($q) use ($words) {
				foreach ($words as $word) {
					$word = str_replace(['\\', '%', '_'], ['\\\\', '\\%', '\\_'], $word);

					$q->where(function ($qq) use ($word) {
						$qq->where('nom_ar', 'like', "%{$word}%")
						   ->orWhere('nom_fr', 'like', "%{$word}%")
						   ->orWhere('nom_en', 'like', "%{$word}%")
						   ->orWhere('nom_directeur', 'like', "%{$word}%")
						   ->orWhere('adresse_ar', 'like', "%{$word}%")
						   ->orWhere('adresse_en', 'like', "%{$word}%")
						   ->orWhere('adresse_fr', 'like', "%{$word}%");
					});
				}
			});
		}

		$auberges = $query->orderBy('nom_ar')->paginate(10)->withQueryString();

		return view('backoffice.auberges.creation_centre.index', compact('auberges', 'search'));
	}

    public function create()
    {
        $milieux = MilieuAuberge::all();
        $gouvernorats = Gouvernorat::all();
        $sports = TpSportsAuberge::all();
        $services = TpServiceAuberge::all();
        $typesChambres = TypeChambre::all();
        $typeTarif = TpTypeTarif::all();

        return view('backoffice.auberges.creation_centre.create', compact('milieux','typeTarif','gouvernorats','sports','services','typesChambres'));
    }
	
	
public function store(Request $request)
{
    $request->validate([
        'nom_ar' => 'required|string|max:100',
        'nom_fr' => 'required|string|max:100',
        'nom_en' => 'required|string|max:100',
        'contact_tel' => 'required|string|max:50',
        'fax' => 'required|string|max:12',
        'contact_email' => 'required|email|max:255',
        'gouvernorat_id' => 'required|integer',
        'delegation_id' => 'required|integer',
        'epa' => 'required|string|in:EPA,Non EPA',
        'id_milieu_auberge' => 'required|integer',
        'adresse_ar' => 'required|string|max:200',
		'adresse_fr' => 'required|string|max:200',
		'adresse_en' => 'required|string|max:200',
		
        'code_postal' => 'required|string|max:20',
        'nom_directeur' => 'required|string|max:255',
        'tel_directeur' => 'required|string|max:50',
        'espace_boisson' => 'required|boolean',
        'piscine' => 'required|boolean',

        'cuisine_fonctionnelle' => 'sometimes|boolean',
        'capacite_batiment' => 'required|integer',
        'nb_chambres' => 'required|integer',
        'nb_chambres_single' => 'nullable|integer',
        'nb_chambres_double' => 'nullable|integer',
        'nb_chambres_triple' => 'nullable|integer',
        'nb_chambres_quadra' => 'nullable|integer',
        'nb_chambres_plus' => 'nullable|integer',
        'nb_salles_de_bain' => 'nullable|integer',
        'nb_salles_de_bain_comm' => 'nullable|integer',
        'nb_chambres_tv' => 'nullable|integer',
        'nb_chambres_clim' => 'nullable|integer',
        'nb_chambres_frigo' => 'nullable|integer',
        'capacite_tentes' => 'nullable|integer',

        'lieux_culturels' => 'nullable|string',
        'autres_services' => 'nullable|string',

        'lieux_culturels_fr' => 'nullable|string',
        'autres_services_fr' => 'nullable|string',

        'lieux_culturels_en' => 'nullable|string',
        'autres_services_en' => 'nullable|string',

        'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
        'description_ar' => 'required|string|max:2000',
        'description_fr' => 'required|string|max:2000',
        'description_en' => 'required|string|max:2000',
        'prix_lit' => 'nullable|numeric|min:0',
        'restaurations' => 'nullable|array|min:1',
        'restaurations.*.nom_fr' => 'nullable|string|max:255',
        'restaurations.*.nom_ar' => 'nullable|string|max:255',
        'restaurations.*.nom_en' => 'nullable|string|max:255',
        'restaurations.*.description_fr' => 'nullable|string',
        'restaurations.*.description_ar' => 'nullable|string',
        'restaurations.*.description_en' => 'nullable|string',
        'restaurations.*.tarif' => 'nullable|numeric|min:0',
        'chambres' => 'nullable|array|min:1',
        'chambres.*.type_chambre_id' => 'required|integer|exists:tp_type_chambre,id_type',
        'chambres.*.capacite' => 'required|integer|min:1',
        'chambres.*.num_chambre' => 'required|string|max:11',

        'espaces' => 'nullable|array|min:1',
        'espaces.*.nom_ar' => 'nullable|string|max:150',
        'espaces.*.nom_fr' => 'nullable|string|max:150',
        'espaces.*.nom_en' => 'nullable|string|max:150',

        'terrains' => 'nullable|array|min:1',
        'terrains.*.nom_ar' => 'nullable|string|max:150',
        'terrains.*.nom_fr' => 'nullable|string|max:150',
        'terrains.*.nom_en' => 'nullable|string|max:150',

        'services_auberge' => 'nullable|array|min:1',
        'services_auberge.*.nom_ar' => 'nullable|string|max:150',
        'services_auberge.*.nom_fr' => 'nullable|string|max:150',
        'services_auberge.*.nom_en' => 'nullable|string|max:150',

        'latitude' => ['nullable', 'numeric', 'between:0,99', 'regex:/^\d{1,2}(\.\d{4})?$/'],
        'longitude' => ['nullable', 'numeric', 'between:0,99', 'regex:/^\d{1,2}(\.\d{4})?$/'],

        'x' => ['nullable', 'numeric', 'between:0,99', 'regex:/^\d{1,2}(\.\d{4})?$/'],
        'y' => ['nullable', 'numeric', 'between:0,99', 'regex:/^\d{1,2}(\.\d{4})?$/'],
		
		
		'map_link' => [
    'required',
    'string',
    'max:500',
    function ($attribute, $value, $fail) {
        // Nettoyer l'URL - enlever les attributs HTML après l'URL
        $value = trim($value);
        
        // Extraire seulement l'URL (avant le premier espace ou guillemet)
        if (preg_match('/^(https?:\/\/[^\s"\']+)/i', $value, $matches)) {
            $value = $matches[1];
        }
        
        // Vérifier si c'est une URL valide
        if (!filter_var($value, FILTER_VALIDATE_URL)) {
            $fail('يجب أن يكون رابط صحيح.');
            return;
        }
        
        $value = strtolower($value);
        
        // Patterns pour liens Google Maps standards
        $standardPatterns = [
            '/^(https?:\/\/)?(www\.)?google\.[a-z]{2,3}(\.[a-z]{2})?\/maps\//',
            '/^(https?:\/\/)?(maps\.google\.[a-z]{2,3}(\.[a-z]{2})?\/)/',
            '/^(https?:\/\/)?goo\.gl\/maps\//',
            '/^(https?:\/\/)?maps\.app\.goo\.gl\//',
            '/^(https?:\/\/)?(www\.)?google\.[a-z]{2,3}(\.[a-z]{2})?\/maps\?q=/',
            '/^(https?:\/\/)?(www\.)?google\.[a-z]{2,3}(\.[a-z]{2})?\/maps\/dir\//',
        ];
        
        // Pattern pour liens embed
        $embedPattern = '/^(https?:\/\/)?(www\.)?(maps\.)?google\.[a-z]{2,3}(\.[a-z]{2})?\/maps\/embed/';
        
        $isValid = false;
        
        // Vérifier les patterns standards
        foreach ($standardPatterns as $pattern) {
            if (preg_match($pattern, $value)) {
                $isValid = true;
                break;
            }
        }
        
        // Vérifier le pattern embed
        if (!$isValid && preg_match($embedPattern, $value)) {
            $isValid = true;
        }
        
        // Vérification par mots-clés
        if (!$isValid) {
            $keywords = [
                'google.com/maps',
                'maps.google.com',
                'goo.gl/maps',
                'maps.app.goo.gl',
                'google.fr/maps',
                'google.tn/maps',
                'google.co.uk/maps',
                'google.de/maps',
                'google.es/maps',
                'google.it/maps',
                'google.ma/maps',
            ];
            
            foreach ($keywords as $keyword) {
                if (strpos($value, $keyword) !== false) {
                    $isValid = true;
                    break;
                }
            }
        }
        
        if (!$isValid) {
            $fail('يجب أن يكون الرابط من Google Maps. مثال: https://www.google.com/maps/place/... أو https://www.google.com/maps/embed?...');
        }
    },
],
		
		
        'tarif_camping_cars' => 'nullable|numeric|min:0',

        'tarifs' => 'nullable|array|min:1',
        'tarifs.*.type' => 'required|in:groupe,single',
        'tarifs.*.id_type_tarif' => 'required|integer|exists:tp_type_tarif,id_type',
        'tarifs.*.tarif_chambre' => 'nullable|numeric|min:0',
        'tarifs.*.tarif_tente' => 'nullable|numeric|min:0',
        'tarifs.*.remarque' => 'nullable|string|max:150',

    ],[
        'nom_ar.required' => 'الرجاء إدخال الاسم بالعربية.',
        'nom_fr.required' => 'الرجاء إدخال الاسم بالفرنسية.',
        'nom_en.required' => 'الرجاء إدخال الاسم بالإنجليزية.',

        'contact_tel.required' => 'الرجاء إدخال رقم الهاتف.',
        'fax.required' => 'الرجاء إدخال رقم الفاكس .',
        'contact_email.required' => 'الرجاء إدخال البريد الإلكتروني.',

        'gouvernorat_id.required' => 'الرجاء اختيار الولاية.',
        'delegation_id.required' => 'الرجاء اختيار المعتمدية.',

        'epa.required' => 'الرجاء تحديد الحالة (EPA أو Non EPA).',

        'id_milieu_auberge.required' => 'الرجاء اختيار وسط النزل.',

        'adresse.required' => 'الرجاء إدخال العنوان.',
        'code_postal.required' => 'الرجاء إدخال الرمز البريدي.',

        'nom_directeur.required' => 'الرجاء إدخال اسم المدير.',
        'tel_directeur.required' => 'الرجاء إدخال رقم هاتف المدير.',

        'espace_boisson.required' => 'الرجاء تحديد توفر فضاء المشروبات.',
        'piscine.required' => 'الرجاء تحديد توفر المسبح.',

        'capacite_batiment.required' => 'الرجاء إدخال طاقة استيعاب المبنى.',

        'nb_chambres.required' => 'الرجاء إدخال عدد الغرف.',

        'image.required' => 'الرجاء تحميل صورة.',
        'image.image' => 'الملف يجب أن يكون صورة.',
        'image.mimes' => 'يجب أن تكون صيغة الصورة: jpeg أو png أو jpg أو gif أو webp.',
        'image.max' => 'حجم الصورة كبير جدًا، الحد الأقصى 2 ميغابايت.',

        'description_ar.required' => 'الرجاء إدخال الوصف بالعربية.',
        'description_fr.required' => 'الرجاء إدخال الوصف بالفرنسية.',
        'description_en.required' => 'الرجاء إدخال الوصف بالإنجليزية.',

        'latitude.required' => 'الرجاء إدخال خط العرض.',
        'longitude.required' => 'الرجاء إدخال خط الطول.',

        'chambres.required' => 'الرجاء إضافة غرفة واحدة على الأقل.',
        'chambres.*.type_chambre_id.required' => 'الرجاء اختيار نوع الغرفة.',
        'chambres.*.capacite.required' => 'الرجاء إدخال طاقة الغرفة.',
        'chambres.*.num_chambre.required' => 'الرجاء إدخال رقم الغرفة.',
		'map_link.required' => 'رابط موقع المركز على الخريطة مطلوب.',
		'map_link.max' => 'رابط موقع المركز يجب ألا يتجاوز 200 حرف.',
		'map_link.url' => 'يجب أن يكون رابط صحيح.',
		'map_link.regex' => 'يجب أن يكون الرابط من Google Maps.',
        'x.required' => 'الرجاء إدخال خط العرض.',
        'y.required' => 'الرجاء إدخال خط الطول.',
    ]);

    // Candidate directories to try for saving uploaded images (ordered).
    $candidates = [
        '/www/mjs/shared_uploads/images',
        '/home/preprov/www/mjs/shared_uploads/images',
        base_path('shared_uploads/images'),
        base_path('../shared_uploads/images'),
        public_path('../../shared_uploads/images'),
    ];

    $usable = null;
    $tried = [];
    $imagePath = null;

    if ($request->hasFile('image')) {
        $file = $request->file('image');

        foreach ($candidates as $cand) {
            $rp = @realpath($cand);
            $tried[$cand] = [
                'realpath' => $rp ?: null,
                'exists' => @file_exists($cand) ? 'yes' : 'no',
                'is_dir' => @is_dir($cand) ? 'yes' : 'no',
                'is_writable' => @is_writable($cand) ? 'yes' : 'no'
            ];
            if ($rp && is_dir($rp) && is_writable($rp)) {
                $usable = $rp;
                break;
            }
        }

        if (!$usable) {
            $diag = [
                'cwd' => @getcwd(),
                'open_basedir' => ini_get('open_basedir') ?: '(not set)',
                'tried' => $tried,
                'get_current_user' => @get_current_user(),
            ];
            \Log::warning('No usable upload folder found for auberge image upload.', $diag);

            $msg = "Upload folder not accessible by PHP. Checked these candidates: ";
            foreach ($tried as $c => $info) {
                $msg .= "[{$c} -> realpath=" . ($info['realpath'] ?? 'null') . ", exists={$info['exists']}, is_dir={$info['is_dir']}, writable={$info['is_writable']}] ";
            }
            return back()->withInput()->withErrors(['image' => $msg]);
        }

        // sanitize original filename
        $originalName = $file->getClientOriginalName();
        $originalName = basename($originalName); // strip any path
        $originalName = preg_replace('/[\0\/\\\\<>:"|?*\x00-\x1F]+/u', '-', $originalName);
        if ($originalName === '' || $originalName === '.' || $originalName === '..') {
            return back()->withInput()->withErrors(['image' => 'اسم الملف الأصلي غير صالح.']);
        }

        $targetPath = rtrim($usable, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $originalName;

        // avoid overwriting: append timestamp if exists
        if (file_exists($targetPath)) {
            $dotPos = strrpos($originalName, '.');
            if ($dotPos !== false) {
                $base = substr($originalName, 0, $dotPos);
                $ext  = substr($originalName, $dotPos);
            } else {
                $base = $originalName;
                $ext = '';
            }
            $originalName = $base . '-' . time() . $ext;
            $targetPath = rtrim($usable, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $originalName;
        }

        try {
            // move the uploaded file to the chosen directory
            $file->move($usable, $originalName);

            // store the absolute path in the DB (e.g. /www/mjs/shared_uploads/images/filename.png)
            $imagePath = $targetPath;
        } catch (\Throwable $e) {
            \Log::error('Auberge image move failed: ' . $e->getMessage(), ['exception' => $e, 'usable' => $usable]);
            return back()->withInput()->withErrors(['image' => 'فشل حفظ الصورة على الخادم. (راجع سجلات الخادم).']);
        }
    }

    $auberge = Auberge::create([
        'nom_ar' => $request->nom_ar,
        'nom_fr' => $request->nom_fr,
        'nom_en' => $request->nom_en,
        'contact_tel' => $request->contact_tel,
        'contact_email' => $request->contact_email,
        'gouver_id' => $request->gouvernorat_id,
        'delegation_id' => $request->delegation_id,
        'epa' => $request->epa,
        'id_milieu_auberge' => $request->id_milieu_auberge,
        'fax' => $request->fax,
        'code_postal' => $request->code_postal,
        'adresse_ar' => $request->adresse_ar,
		'adresse_fr' => $request->adresse_fr,
		'adresse_en' => $request->adresse_en,
        'nom_directeur' => $request->nom_directeur,
        'tel_directeur' => $request->tel_directeur,
        'espace_boisson' => $request->has('espace_boisson') ? 1 : 0,
        'piscine' => $request->has('piscine') ? 1 : 0,

        'capacite_batiment' => $request->capacite_batiment,
        'nb_chambres' => $request->nb_chambres,
        'nb_chambres_single' => $request->nb_chambres_single,
        'nb_chambres_double' => $request->nb_chambres_double,
        'nb_chambres_triple' => $request->nb_chambres_triple,
        'nb_chambres_quadra' => $request->nb_chambres_quadra,
        'nb_chambres_plus' => $request->nb_chambres_plus,
        'nb_salles_de_bain' => $request->nb_salles_de_bain,
        'nb_salles_de_bain_comm' => $request->nb_salles_de_bain_comm,
        'nb_chambres_tv' => $request->nb_chambres_tv,
        'nb_chambres_clim' => $request->nb_chambres_clim,
        'nb_chambres_frigo' => $request->nb_chambres_frigo,
        'capacite_tentes' => $request->capacite_tentes,
        'cuisine_fonct' => $request->has('cuisine_fonctionnelle') ? 1 : 0,
        'lieux_culturels' => $request->lieux_culturels,
        'autres_services' => $request->autres_services,

        'lieux_culturels_fr' => $request->lieux_culturels_fr,
        'autres_services_fr' => $request->autres_services_fr,

        'lieux_culturels_en' => $request->lieux_culturels_en,
        'autres_services_en' => $request->autres_services_en,


        'description_ar' => $request->description_ar,
        'description_fr' => $request->description_fr,
        'description_en' => $request->description_en,
        'latitude' => $request->latitude,
        'longitude' => $request->longitude,
        'prix_lit'=> $request->prix_lit,
        'image' => $imagePath,
        'tarif_camping_cars' => $request->tarif_camping_cars,
        'x' => $request->x,
        'y' => $request->y,
        'map_link' => $request->map_link,
    ]);

    if ($request->has('restaurations')) {
        foreach ($request->restaurations as $restaurationData) {
            RestaurationAuberge::create([
                'auberge_id' => $auberge->id,
                'nom_fr' => $restaurationData['nom_fr'],
                'nom_ar' => $restaurationData['nom_ar'],
                'nom_en' => $restaurationData['nom_en'],
                'description_fr' => $restaurationData['description_fr'] ?? null,
                'description_ar' => $restaurationData['description_ar'] ?? null,
                'description_en' => $restaurationData['description_en'] ?? null,
                'tarif' => $restaurationData['tarif'],
            ]);
        }
    }
    if ($request->has('sports')) {
        $auberge->sports()->sync($request->sports);
    }
    if ($request->has('services')) {
        $auberge->services()->sync($request->services);
    }

    if ($request->has('chambres')) {
        foreach ($request->chambres as $chambreData) {
            ChambreTente::create([
                'auberge_id' => $auberge->id,
                'type_chambre_id' => $chambreData['type_chambre_id'],
                'capacite' => $chambreData['capacite'],
                'num_chambre' => $chambreData['num_chambre'],
                'type' => 'chambre',
            ]);
        }
    }

    if ($request->has('espaces')) {
        foreach ($request->espaces as $espaceData) {
            EspaceAuberge::create([
                'id_auberge' => $auberge->id,
                'nom_ar' => $espaceData['nom_ar'],
                'nom_fr' => $espaceData['nom_fr'],
                'nom_en' => $espaceData['nom_en'],
            ]);
        }
    }

    if ($request->has('terrains')) {
        foreach ($request->terrains as $terrainData) {
            TerrainSportifAuberge::create([
                'id_auberge' => $auberge->id,
                'nom_ar' => $terrainData['nom_ar'],
                'nom_fr' => $terrainData['nom_fr'],
                'nom_en' => $terrainData['nom_en'],
            ]);
        }
    }

    if ($request->has('services_auberge')) {
        foreach ($request->services_auberge as $serviceData) {
            ServiceAuberge::create([
                'id_auberge' => $auberge->id,
                'nom_ar' => $serviceData['nom_ar'],
                'nom_fr' => $serviceData['nom_fr'],
                'nom_en' => $serviceData['nom_en'],
            ]);
        }
    }

    if ($request->has('tarifs')) {
        foreach ($request->tarifs as $tarifData) {
            TarifAuberge::create([
                'id_auberge' => $auberge->id,
                'type' => $tarifData['type'],
                'id_type_tarif' => $tarifData['id_type_tarif'],
                'tarif_chambre' => $tarifData['tarif_chambre'] ?? 0,
                'tarif_tente' => $tarifData['tarif_tente'] ?? 0,
                'remarque' => $tarifData['remarque'] ?? null,
            ]);
        }
    }

    return redirect()->route('auberges.index')
                    ->with('success', 'تمت إضافة المركب أو مركز الإقامة بنجاح !');
}


  public function edit($id)
{
    $auberge = Auberge::with('espaces','terrains','services_auberge')->findOrFail($id);
    $milieux = MilieuAuberge::all();
    $gouvernorats = Gouvernorat::all();
    $delegations = Delegation::all();
    $sports = TpSportsAuberge::all();
    $services = TpServiceAuberge::all();
    $typesChambres = TypeChambre::all();
    $typeTarif = TpTypeTarif::all();

    // compute image source (data:base64 or URL) for the auberge image
    $computeImageSrc = function ($stored) {
        if (empty($stored)) return null;

        // already a full URL?
        if (filter_var($stored, FILTER_VALIDATE_URL)) {
            return $stored;
        }

        // candidate filesystem paths (ordered)
        $serverRoot = '/www/mjs';
        $candidates = [
            $stored, // as stored (maybe absolute path or relative path)
        ];

        // try prefixing server root if stored is a relative path or filename
        if (strpos($stored, $serverRoot) !== 0) {
            $candidates[] = rtrim($serverRoot, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . ltrim($stored, DIRECTORY_SEPARATOR);
        }

        // fallback to shared uploads images folder using basename
        $candidates[] = rtrim($serverRoot, DIRECTORY_SEPARATOR) . '/shared_uploads/images/' . basename($stored);

        // public path fallback (if value is relative to public/)
        $candidates[] = public_path(ltrim($stored, '/'));

        // storage/app fallback
        $candidates[] = storage_path('app/' . ltrim($stored, '/'));

        // dedupe & normalize
        $seen = [];
        $candidates = array_values(array_filter(array_map(function ($p) use (&$seen) {
            $p = (string)$p;
            if ($p === '') return null;
            // strip file:// if present
            $p = preg_replace('#^file://#', '', $p);
            if (isset($seen[$p])) return null;
            $seen[$p] = true;
            return $p;
        }, $candidates)));

        // try to read a real file
        foreach ($candidates as $path) {
            if (@file_exists($path) && @is_file($path) && @is_readable($path)) {
                $mime = @mime_content_type($path) ?: 'image/jpeg';
                $contents = @file_get_contents($path);
                if ($contents !== false && $contents !== null) {
                    return "data:{$mime};base64," . base64_encode($contents);
                }
            }
        }

        // if the stored value begins with slash, expose it as an absolute URL
        if (strpos($stored, '/') === 0) {
            return url(ltrim($stored, '/'));
        }

        // fallback: stored might be raw binary blob — return it base64-encoded (assume jpeg)
        try {
            return 'data:image/jpeg;base64,' . base64_encode($stored);
        } catch (\Throwable $e) {
            return null;
        }
    };

    $imageSrc = $computeImageSrc($auberge->image);

    return view('backoffice.auberges.creation_centre.edit', compact(
        'auberge','typeTarif','milieux','gouvernorats','sports','services','delegations','typesChambres','imageSrc'
    ));
}


  public function edit2($id)
{
    $auberge = Auberge::with('espaces','terrains','services_auberge')->findOrFail($id);
    $milieux = MilieuAuberge::all();
    $gouvernorats = Gouvernorat::all();
    $delegations = Delegation::all();
    $sports = TpSportsAuberge::all();
    $services = TpServiceAuberge::all();
    $typesChambres = TypeChambre::all();
    $typeTarif = TpTypeTarif::all();

    // compute image source (data:base64 or URL) for the auberge image
    $computeImageSrc = function ($stored) {
        if (empty($stored)) return null;

        // already a full URL?
        if (filter_var($stored, FILTER_VALIDATE_URL)) {
            return $stored;
        }

        // candidate filesystem paths (ordered)
        $serverRoot = '/www/mjs';
        $candidates = [
            $stored, // as stored (maybe absolute path or relative path)
        ];

        // try prefixing server root if stored is a relative path or filename
        if (strpos($stored, $serverRoot) !== 0) {
            $candidates[] = rtrim($serverRoot, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . ltrim($stored, DIRECTORY_SEPARATOR);
        }

        // fallback to shared uploads images folder using basename
        $candidates[] = rtrim($serverRoot, DIRECTORY_SEPARATOR) . '/shared_uploads/images/' . basename($stored);

        // public path fallback (if value is relative to public/)
        $candidates[] = public_path(ltrim($stored, '/'));

        // storage/app fallback
        $candidates[] = storage_path('app/' . ltrim($stored, '/'));

        // dedupe & normalize
        $seen = [];
        $candidates = array_values(array_filter(array_map(function ($p) use (&$seen) {
            $p = (string)$p;
            if ($p === '') return null;
            // strip file:// if present
            $p = preg_replace('#^file://#', '', $p);
            if (isset($seen[$p])) return null;
            $seen[$p] = true;
            return $p;
        }, $candidates)));

        // try to read a real file
        foreach ($candidates as $path) {
            if (@file_exists($path) && @is_file($path) && @is_readable($path)) {
                $mime = @mime_content_type($path) ?: 'image/jpeg';
                $contents = @file_get_contents($path);
                if ($contents !== false && $contents !== null) {
                    return "data:{$mime};base64," . base64_encode($contents);
                }
            }
        }

        // if the stored value begins with slash, expose it as an absolute URL
        if (strpos($stored, '/') === 0) {
            return url(ltrim($stored, '/'));
        }

        // fallback: stored might be raw binary blob — return it base64-encoded (assume jpeg)
        try {
            return 'data:image/jpeg;base64,' . base64_encode($stored);
        } catch (\Throwable $e) {
            return null;
        }
    };

    $imageSrc = $computeImageSrc($auberge->image);

    return view('backoffice.auberges.creation_centre.edit2', compact(
            'auberge','milieux','gouvernorats','typeTarif','sports','services','delegations','typesChambres'
        ));
}


	
/*	public function edit2($id)
    {
        $auberge = Auberge::with('espaces','terrains','services_auberge')->findOrFail($id);
        $milieux = MilieuAuberge::all();
        $gouvernorats = Gouvernorat::all();
        $delegations = Delegation::all();
        $sports = TpSportsAuberge::all();
        $services = TpServiceAuberge::all();
        $typesChambres = TypeChambre::all();
        $typeTarif = TpTypeTarif::all();

        return view('backoffice.auberges.creation_centre.edit2', compact(
            'auberge','milieux','gouvernorats','typeTarif','sports','services','delegations','typesChambres'
        ));
    } */

public function update(Request $request, $id)
{
    $auberge = Auberge::findOrFail($id);

    $request->validate([
        'nom_ar' => 'required|string|max:100',
        'nom_fr' => 'required|string|max:100',
        'nom_en' => 'required|string|max:100',
        'contact_tel' => 'required|string|max:50',
        'fax' => 'required|string|max:12',
        'contact_email' => 'required|email|max:255',
        'gouvernorat_id' => 'required|integer',
        'delegation_id' => 'required|integer',
        'epa' => 'required|string|in:EPA,Non EPA',
        'id_milieu_auberge' => 'required|integer',
        'adresse_ar' => 'required|string|max:200',
		'adresse_fr' => 'required|string|max:200',
		'adresse_en' => 'required|string|max:200',
        'code_postal' => 'required|string|max:20',
        'nom_directeur' => 'required|string|max:255',
        'tel_directeur' => 'required|string|max:50',
        'espace_boisson' => 'required|boolean',
        'piscine' => 'required|boolean',

        'cuisine_fonctionnelle' => 'sometimes|boolean',
        'capacite_batiment' => 'required|integer',
        'nb_chambres' => 'required|integer',
        'nb_chambres_single' => 'nullable|integer',
        'nb_chambres_double' => 'nullable|integer',
        'nb_chambres_triple' => 'nullable|integer',
        'nb_chambres_quadra' => 'nullable|integer',
        'nb_chambres_plus' => 'nullable|integer',
        'nb_salles_de_bain' => 'nullable|integer',
        'nb_salles_de_bain_comm' => 'nullable|integer',
        'nb_chambres_tv' => 'nullable|integer',
        'nb_chambres_clim' => 'nullable|integer',
        'nb_chambres_frigo' => 'nullable|integer',
        'capacite_tentes' => 'nullable|integer',
        'lieux_culturels' => 'nullable|string',
        'autres_services' => 'nullable|string',

        'lieux_culturels_fr' => 'nullable|string',
        'autres_services_fr' => 'nullable|string',


        'lieux_culturels_en' => 'nullable|string',
        'autres_services_en' => 'nullable|string',

'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
        //'image' => $auberge->image ? 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048' : 'required|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
        'description_ar' => 'required|string|max:2000',
        'description_fr' => 'required|string|max:2000',
        'description_en' => 'required|string|max:2000',
        'prix_lit' => 'nullable|numeric|min:0',
        'restaurations' => 'nullable|array|min:1',
        'restaurations.*.nom_fr' => 'nullable|string|max:255',
        'restaurations.*.nom_ar' => 'nullable|string|max:255',
        'restaurations.*.nom_en' => 'nullable|string|max:255',
        'restaurations.*.description_fr' => 'nullable|string',
        'restaurations.*.description_ar' => 'nullable|string',
        'restaurations.*.description_en' => 'nullable|string',
        'restaurations.*.tarif' => 'nullable|numeric|min:0',
        'chambres' => 'nullable|array|min:1',
        'chambres.*.type_chambre_id' => 'required|integer|exists:tp_type_chambre,id_type',
        'chambres.*.capacite' => 'required|integer|min:1',
        'chambres.*.num_chambre' => 'required|string|max:11',

        'espaces' => 'nullable|array|min:1',
        'espaces.*.nom_ar' => 'nullable|string|max:150',
        'espaces.*.nom_fr' => 'nullable|string|max:150',
        'espaces.*.nom_en' => 'nullable|string|max:150',

        'terrains' => 'nullable|array|min:1',
        'terrains.*.nom_ar' => 'nullable|string|max:150',
        'terrains.*.nom_fr' => 'nullable|string|max:150',
        'terrains.*.nom_en' => 'nullable|string|max:150',

        'services_auberge' => 'nullable|array|min:1',
        'services_auberge.*.nom_ar' => 'nullable|string|max:150',
        'services_auberge.*.nom_fr' => 'nullable|string|max:150',
        'services_auberge.*.nom_en' => 'nullable|string|max:150',

        'latitude' => ['nullable', 'numeric', 'between:0,99', 'regex:/^\d{1,2}(\.\d{4})?$/'],
        'longitude' => ['nullable', 'numeric', 'between:0,99', 'regex:/^\d{1,2}(\.\d{4})?$/'],
'map_link' => [
    'required',
    'string',
    'max:500',
    function ($attribute, $value, $fail) {
        // Nettoyer l'URL - enlever les attributs HTML après l'URL
        $value = trim($value);
        
        // Extraire seulement l'URL (avant le premier espace ou guillemet)
        if (preg_match('/^(https?:\/\/[^\s"\']+)/i', $value, $matches)) {
            $value = $matches[1];
        }
        
        // Vérifier si c'est une URL valide
        if (!filter_var($value, FILTER_VALIDATE_URL)) {
            $fail('يجب أن يكون رابط صحيح.');
            return;
        }
        
        $value = strtolower($value);
        
        // Patterns pour liens Google Maps standards
        $standardPatterns = [
            '/^(https?:\/\/)?(www\.)?google\.[a-z]{2,3}(\.[a-z]{2})?\/maps\//',
            '/^(https?:\/\/)?(maps\.google\.[a-z]{2,3}(\.[a-z]{2})?\/)/',
            '/^(https?:\/\/)?goo\.gl\/maps\//',
            '/^(https?:\/\/)?maps\.app\.goo\.gl\//',
            '/^(https?:\/\/)?(www\.)?google\.[a-z]{2,3}(\.[a-z]{2})?\/maps\?q=/',
            '/^(https?:\/\/)?(www\.)?google\.[a-z]{2,3}(\.[a-z]{2})?\/maps\/dir\//',
        ];
        
        // Pattern pour liens embed
        $embedPattern = '/^(https?:\/\/)?(www\.)?(maps\.)?google\.[a-z]{2,3}(\.[a-z]{2})?\/maps\/embed/';
        
        $isValid = false;
        
        // Vérifier les patterns standards
        foreach ($standardPatterns as $pattern) {
            if (preg_match($pattern, $value)) {
                $isValid = true;
                break;
            }
        }
        
        // Vérifier le pattern embed
        if (!$isValid && preg_match($embedPattern, $value)) {
            $isValid = true;
        }
        
        // Vérification par mots-clés
        if (!$isValid) {
            $keywords = [
                'google.com/maps',
                'maps.google.com',
                'goo.gl/maps',
                'maps.app.goo.gl',
                'google.fr/maps',
                'google.tn/maps',
                'google.co.uk/maps',
                'google.de/maps',
                'google.es/maps',
                'google.it/maps',
                'google.ma/maps',
            ];
            
            foreach ($keywords as $keyword) {
                if (strpos($value, $keyword) !== false) {
                    $isValid = true;
                    break;
                }
            }
        }
        
        if (!$isValid) {
            $fail('يجب أن يكون الرابط من Google Maps. مثال: https://www.google.com/maps/place/... أو https://www.google.com/maps/embed?...');
        }
    },
],
	'x' => ['nullable', 'numeric', 'between:0,99', 'regex:/^\d{1,2}(\.\d{4})?$/'],
        'y' => ['nullable', 'numeric', 'between:0,99', 'regex:/^\d{1,2}(\.\d{4})?$/'],
        'tarif_camping_cars' => 'nullable|numeric|min:0',

        'tarifs' => 'nullable|array|min:1',
        'tarifs.*.type' => 'required|in:groupe,single',
        'tarifs.*.id_type_tarif' => 'required|integer|exists:tp_type_tarif,id_type',
        'tarifs.*.tarif_chambre' => 'nullable|numeric|min:0',
        'tarifs.*.tarif_tente' => 'nullable|numeric|min:0',
        'tarifs.*.remarque' => 'nullable|string|max:150',

    ],[
        'nom_ar.required' => 'الرجاء إدخال الاسم بالعربية.',
        'nom_fr.required' => 'الرجاء إدخال الاسم بالفرنسية.',
        'nom_en.required' => 'الرجاء إدخال الاسم بالإنجليزية.',
'map_link.required' => 'رابط موقع المركز على الخريطة مطلوب.',
    'map_link.max' => 'رابط موقع المركز يجب ألا يتجاوز 200 حرف.',
    'map_link.url' => 'يجب أن يكون رابط صحيح.',
    'map_link.regex' => 'يجب أن يكون الرابط من Google Maps.',
        'contact_tel.required' => 'الرجاء إدخال رقم الهاتف.',
        'fax.required' => 'الرجاء إدخال رقم الفاكس .',
        'contact_email.required' => 'الرجاء إدخال البريد الإلكتروني.',

        'gouvernorat_id.required' => 'الرجاء اختيار الولاية.',
        'delegation_id.required' => 'الرجاء اختيار المعتمدية.',

        'epa.required' => 'الرجاء تحديد الحالة (EPA أو Non EPA).',

        'id_milieu_auberge.required' => 'الرجاء اختيار وسط النزل.',

        'adresse.required' => 'الرجاء إدخال العنوان.',
        'code_postal.required' => 'الرجاء إدخال الرمز البريدي.',

        'nom_directeur.required' => 'الرجاء إدخال اسم المدير.',
        'tel_directeur.required' => 'الرجاء إدخال رقم هاتف المدير.',

        'espace_boisson.required' => 'الرجاء تحديد توفر فضاء المشروبات.',
        'piscine.required' => 'الرجاء تحديد توفر المسبح.',

        'capacite_batiment.required' => 'الرجاء إدخال طاقة استيعاب المبنى.',

        'nb_chambres.required' => 'الرجاء إدخال عدد الغرف.',

        'image.required' => 'الرجاء تحميل صورة.',
        'image.image' => 'الملف يجب أن يكون صورة.',
        'image.mimes' => 'يجب أن تكون صيغة الصورة: jpeg أو png أو jpg أو gif أو webp.',
        'image.max' => 'حجم الصورة كبير جدًا، الحد الأقصى 2 ميغابايت.',

        'description_ar.required' => 'الرجاء إدخال الوصف بالعربية.',
        'description_fr.required' => 'الرجاء إدخال الوصف بالفرنسية.',
        'description_en.required' => 'الرجاء إدخال الوصف بالالإنجليزية.',

        'latitude.required' => 'الرجاء إدخال خط العرض.',
        'longitude.required' => 'الرجاء إدخال خط الطول.',

        'chambres.required' => 'الرجاء إضافة غرفة واحدة على الأقل.',
        'chambres.*.type_chambre_id.required' => 'الرجاء اختيار نوع الغرفة.',
        'chambres.*.capacite.required' => 'الرجاء إدخال طاقة الغرفة.',
        'chambres.*.num_chambre.required' => 'الرجاء إدخال رقم الغرفة.',

        'x.required' => 'الرجاء إدخال خط العرض.',
        'y.required' => 'الرجاء إدخال خط الطول.',
    ]);

    // Candidate directories to try for saving uploaded images (ordered).
    $candidates = [
        '/www/mjs/shared_uploads/images',
        '/home/preprov/www/mjs/shared_uploads/images',
        base_path('shared_uploads/images'),
        base_path('../shared_uploads/images'),
        public_path('../../shared_uploads/images'),
    ];

    $usable = null;
    $tried = [];
    $imagePath = null;

    // If an image was uploaded, try to move it to a usable shared folder and set $imagePath
    if ($request->hasFile('image')) {
        $file = $request->file('image');

        foreach ($candidates as $cand) {
            $rp = @realpath($cand);
            $tried[$cand] = [
                'realpath' => $rp ?: null,
                'exists' => @file_exists($cand) ? 'yes' : 'no',
                'is_dir' => @is_dir($cand) ? 'yes' : 'no',
                'is_writable' => @is_writable($cand) ? 'yes' : 'no'
            ];
            if ($rp && is_dir($rp) && is_writable($rp)) {
                $usable = $rp;
                break;
            }
        }

        if (!$usable) {
            $diag = [
                'cwd' => @getcwd(),
                'open_basedir' => ini_get('open_basedir') ?: '(not set)',
                'tried' => $tried,
                'get_current_user' => @get_current_user(),
            ];
            \Log::warning('No usable upload folder found for auberge image upload (update).', $diag);

            $msg = "Upload folder not accessible by PHP. Checked these candidates: ";
            foreach ($tried as $c => $info) {
                $msg .= "[{$c} -> realpath=" . ($info['realpath'] ?? 'null') . ", exists={$info['exists']}, is_dir={$info['is_dir']}, writable={$info['is_writable']}] ";
            }
            return back()->withInput()->withErrors(['image' => $msg]);
        }

        // sanitize original filename
        $originalName = $file->getClientOriginalName();
        $originalName = basename($originalName);
        $originalName = preg_replace('/[\0\/\\\\<>:"|?*\x00-\x1F]+/u', '-', $originalName);
        if ($originalName === '' || $originalName === '.' || $originalName === '..') {
            return back()->withInput()->withErrors(['image' => 'اسم الملف الأصلي غير صالح.']);
        }

        $targetPath = rtrim($usable, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $originalName;

        // avoid overwriting: append timestamp if exists
        if (file_exists($targetPath)) {
            $dotPos = strrpos($originalName, '.');
            if ($dotPos !== false) {
                $base = substr($originalName, 0, $dotPos);
                $ext  = substr($originalName, $dotPos);
            } else {
                $base = $originalName;
                $ext = '';
            }
            $originalName = $base . '-' . time() . $ext;
            $targetPath = rtrim($usable, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $originalName;
        }

        try {
            $file->move($usable, $originalName);
            $imagePath = $targetPath;
        } catch (\Throwable $e) {
            \Log::error('Auberge image move failed (update): ' . $e->getMessage(), ['exception' => $e, 'usable' => $usable]);
            return back()->withInput()->withErrors(['image' => 'فشل حفظ الصورة على الخادم. (راجع سجلات الخادم).']);
        }
    }

    // Prepare update data (preserve existing image if no new one uploaded)
    $updateData = [
        'nom_ar' => $request->nom_ar,
        'nom_fr' => $request->nom_fr,
        'nom_en' => $request->nom_en,
        'contact_tel' => $request->contact_tel,
        'contact_email' => $request->contact_email,
        'gouver_id' => $request->gouvernorat_id,
        'delegation_id' => $request->delegation_id,
        'epa' => $request->epa,
        'id_milieu_auberge' => $request->id_milieu_auberge,
        'fax' => $request->fax,
        'code_postal' => $request->code_postal,
        'adresse_ar' => $request->adresse_ar,
		'adresse_fr' => $request->adresse_fr,
		'adresse_en' => $request->adresse_en,
        'nom_directeur' => $request->nom_directeur,
        'tel_directeur' => $request->tel_directeur,
        'espace_boisson' => $request->has('espace_boisson') ? 1 : 0,
        'piscine' => $request->has('piscine') ? 1 : 0,

        'capacite_batiment' => $request->capacite_batiment,
        'nb_chambres' => $request->nb_chambres,
        'nb_chambres_single' => $request->nb_chambres_single,
        'nb_chambres_double' => $request->nb_chambres_double,
        'nb_chambres_triple' => $request->nb_chambres_triple,
        'nb_chambres_quadra' => $request->nb_chambres_quadra,
        'nb_chambres_plus' => $request->nb_chambres_plus,
        'nb_salles_de_bain' => $request->nb_salles_de_bain,
        'nb_salles_de_bain_comm' => $request->nb_salles_de_bain_comm,
        'nb_chambres_tv' => $request->nb_chambres_tv,
        'nb_chambres_clim' => $request->nb_chambres_clim,
        'nb_chambres_frigo' => $request->nb_chambres_frigo,
        'capacite_tentes' => $request->capacite_tentes,
        'cuisine_fonct' => $request->has('cuisine_fonctionnelle') ? 1 : 0,
        'lieux_culturels' => $request->lieux_culturels,
        'autres_services' => $request->autres_services,

        'lieux_culturels_fr' => $request->lieux_culturels_en,
        'autres_services_fr' => $request->autres_services_en,

        'lieux_culturels_en' => $request->lieux_culturels_en,
        'autres_services_en' => $request->autres_services_en,

        'description_ar' => $request->description_ar,
        'description_fr' => $request->description_fr,
        'description_en' => $request->description_en,
        'tarif_camping_cars' => $request->tarif_camping_cars,
        'latitude' => $request->latitude,
        'longitude' => $request->longitude,
        'prix_lit'=> $request->prix_lit,
        'x' => $request->x,
        'y' => $request->y,
        'map_link' => $request->map_link,
    ];

    // set image field to new path if uploaded, otherwise leave existing value untouched
    if ($imagePath !== null) {
        $updateData['image'] = $imagePath;
    }

    $auberge->update($updateData);

    // continue with the rest of your existing logic (restaurations, chambres, relations, etc.)
    if ($request->has('restaurations')) {
        foreach ($request->restaurations as $restaurationData) {
            if (!empty($restaurationData['id_rest'])) {
                RestaurationAuberge::where('id_rest', $restaurationData['id_rest'])
                    ->update([
                        'nom_fr' => $restaurationData['nom_fr'],
                        'nom_ar' => $restaurationData['nom_ar'],
                        'nom_en' => $restaurationData['nom_en'],
                        'description_fr' => $restaurationData['description_fr'] ?? null,
                        'description_ar' => $restaurationData['description_ar'] ?? null,
                        'description_en' => $restaurationData['description_en'] ?? null,
                        'tarif' => $restaurationData['tarif'],
                    ]);
            } else {
                RestaurationAuberge::create([
                    'auberge_id' => $auberge->id,
                    'nom_fr' => $restaurationData['nom_fr'],
                    'nom_ar' => $restaurationData['nom_ar'],
                    'nom_en' => $restaurationData['nom_en'],
                    'description_fr' => $restaurationData['description_fr'] ?? null,
                    'description_ar' => $restaurationData['description_ar'] ?? null,
                    'description_en' => $restaurationData['description_en'] ?? null,
                    'tarif' => $restaurationData['tarif'],
                ]);
            }
        }
    }

    if ($request->has('chambres')) {
        foreach ($request->chambres as $ch) {
            if (!empty($ch['id_chambre_tente'])) {
                chambreTente::where('id_chambre_tente', $ch['id_chambre_tente'])
                    ->update([
                        'type_chambre_id' => $ch['type_chambre_id'],
                        'capacite' => $ch['capacite'],
                        'num_chambre' => $ch['num_chambre'],
                    ]);
            } else {
                ChambreTente::create([
                    'auberge_id' => $auberge->id,
                    'type_chambre_id' => $ch['type_chambre_id'],
                    'capacite' => $ch['capacite'],
                    'num_chambre' => $ch['num_chambre'],
                    'type' => 'chambre',
                ]);
            }
        }
    }

    if ($request->has('sports')) {
        $auberge->sports()->sync($request->sports);
    } else {
        $auberge->sports()->sync([]);
    }

    if ($request->has('services')) {
        $auberge->services()->sync($request->services);
    } else {
        $auberge->services()->sync([]);
    }

    if ($request->has('espaces')) {
        foreach ($request->espaces as $espaceData) {
            if (!empty($espaceData['id_espace'])) {
                EspaceAuberge::where('id_espace', $espaceData['id_espace'])->update([
                    'nom_ar' => $espaceData['nom_ar'],
                    'nom_fr' => $espaceData['nom_fr'],
                    'nom_en' => $espaceData['nom_en'],
                    'id_auberge' => $auberge->id,
                ]);
            } else {
                EspaceAuberge::create([
                    'nom_ar' => $espaceData['nom_ar'],
                    'nom_fr' => $espaceData['nom_fr'],
                    'nom_en' => $espaceData['nom_en'],
                    'id_auberge' => $auberge->id,
                ]);
            }
        }
    }

    if ($request->has('terrains')) {
        foreach ($request->terrains as $terrainsData) {
            if (!empty($terrainsData['id_terrain'])) {
                TerrainSportifAuberge::where('id_terrain', $terrainsData['id_terrain'])->update([
                    'nom_ar' => $terrainsData['nom_ar'],
                    'nom_fr' => $terrainsData['nom_fr'],
                    'nom_en' => $terrainsData['nom_en'],
                    'id_auberge' => $auberge->id,
                ]);
            } else {
                TerrainSportifAuberge::create([
                    'nom_ar' => $terrainsData['nom_ar'],
                    'nom_fr' => $terrainsData['nom_fr'],
                    'nom_en' => $terrainsData['nom_en'],
                    'id_auberge' => $auberge->id,
                ]);
            }
        }
    }

    if ($request->has('services_auberge')) {
        foreach ($request->services_auberge as $services_aubergeData) {
            if (!empty($services_aubergeData['id_service'])) {
                ServiceAuberge::where('id_service', $services_aubergeData['id_service'])->update([
                    'nom_ar' => $services_aubergeData['nom_ar'],
                    'nom_fr' => $services_aubergeData['nom_fr'],
                    'nom_en' => $services_aubergeData['nom_en'],
                    'id_auberge' => $auberge->id,
                ]);
            } else {
                ServiceAuberge::create([
                    'nom_ar' => $services_aubergeData['nom_ar'],
                    'nom_fr' => $services_aubergeData['nom_fr'],
                    'nom_en' => $services_aubergeData['nom_en'],
                    'id_auberge' => $auberge->id,
                ]);
            }
        }
    }

    $auberge->tarifs()->delete();

    if ($request->has('tarifs')) {
        foreach ($request->tarifs as $tarifData) {
            TarifAuberge::create([
                'id_auberge' => $auberge->id,
                'type' => $tarifData['type'],
                'id_type_tarif' => $tarifData['id_type_tarif'],
                'tarif_chambre' => $tarifData['tarif_chambre'] ?? 0,
                'tarif_tente' => $tarifData['tarif_tente'] ?? 0,
                'remarque' => $tarifData['remarque'] ?? null,
            ]);
        }
    }

    return redirect()->route('auberges.index')
                    ->with('success', 'تم تحديث بيانات المركب أو مركز الإقامة بنجاح !');
}

	
	
	public function update2(Request $request, $id)
{
    $auberge = Auberge::findOrFail($id);

    $request->validate([
        'nom_ar' => 'required|string|max:100',
        'nom_fr' => 'required|string|max:100',
        'nom_en' => 'required|string|max:100',
        'contact_tel' => 'required|string|max:50',
        'fax' => 'required|string|max:12',
        'contact_email' => 'required|email|max:255',
        'gouvernorat_id' => 'required|integer',
        'delegation_id' => 'required|integer',
        'epa' => 'required|string|in:EPA,Non EPA',
        'id_milieu_auberge' => 'required|integer',
        'adresse_ar' => 'required|string|max:200',
		'adresse_fr' => 'required|string|max:200',
		'adresse_en' => 'required|string|max:200',
        'code_postal' => 'required|string|max:20',
        'nom_directeur' => 'required|string|max:255',
        'tel_directeur' => 'required|string|max:50',
        'espace_boisson' => 'required|boolean',
        'piscine' => 'required|boolean',

        'cuisine_fonctionnelle' => 'sometimes|boolean',
        'capacite_batiment' => 'required|integer',
        'nb_chambres' => 'required|integer',
        'nb_chambres_single' => 'nullable|integer',
        'nb_chambres_double' => 'nullable|integer',
        'nb_chambres_triple' => 'nullable|integer',
        'nb_chambres_quadra' => 'nullable|integer',
        'nb_chambres_plus' => 'nullable|integer',
        'nb_salles_de_bain' => 'nullable|integer',
        'nb_salles_de_bain_comm' => 'nullable|integer',
        'nb_chambres_tv' => 'nullable|integer',
        'nb_chambres_clim' => 'nullable|integer',
        'nb_chambres_frigo' => 'nullable|integer',
        'capacite_tentes' => 'nullable|integer',
        'lieux_culturels' => 'nullable|string',
        'autres_services' => 'nullable|string',
        //'image' => $auberge->image ? 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048' : 'required|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
        'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
		'description_ar' => 'required|string|max:2000',
        'description_fr' => 'required|string|max:2000',
        'description_en' => 'required|string|max:2000',
        'prix_lit' => 'nullable|numeric|min:0',
        'restaurations' => 'nullable|array|min:1',
        'restaurations.*.nom_fr' => 'nullable|string|max:255',
        'restaurations.*.nom_ar' => 'nullable|string|max:255',
        'restaurations.*.nom_en' => 'nullable|string|max:255',
        'restaurations.*.description_fr' => 'nullable|string',
        'restaurations.*.description_ar' => 'nullable|string',
        'restaurations.*.description_en' => 'nullable|string',
        'restaurations.*.tarif' => 'nullable|numeric|min:0',
        'chambres' => 'nullable|array|min:1',
        'chambres.*.type_chambre_id' => 'required|integer|exists:tp_type_chambre,id_type',
        'chambres.*.capacite' => 'required|integer|min:1',
        'chambres.*.num_chambre' => 'required|string|max:11',

        'espaces' => 'nullable|array|min:1',
        'espaces.*.nom_ar' => 'nullable|string|max:150',
        'espaces.*.nom_fr' => 'nullable|string|max:150',
        'espaces.*.nom_en' => 'nullable|string|max:150',

        'terrains' => 'nullable|array|min:1',
        'terrains.*.nom_ar' => 'nullable|string|max:150',
        'terrains.*.nom_fr' => 'nullable|string|max:150',
        'terrains.*.nom_en' => 'nullable|string|max:150',

        'services_auberge' => 'nullable|array|min:1',
        'services_auberge.*.nom_ar' => 'nullable|string|max:150',
        'services_auberge.*.nom_fr' => 'nullable|string|max:150',
        'services_auberge.*.nom_en' => 'nullable|string|max:150',

        'latitude' => ['nullable', 'numeric', 'between:0,99', 'regex:/^\d{1,2}(\.\d{4})?$/'],
        'longitude' => ['nullable', 'numeric', 'between:0,99', 'regex:/^\d{1,2}(\.\d{4})?$/'],

        'x' => ['nullable', 'numeric', 'between:0,99', 'regex:/^\d{1,2}(\.\d{4})?$/'],
        'y' => ['nullable', 'numeric', 'between:0,99', 'regex:/^\d{1,2}(\.\d{4})?$/'],
        'tarif_camping_cars' => 'nullable|numeric|min:0',

		'map_link' => [
    'required',
    'string',
    'max:500',
    function ($attribute, $value, $fail) {
        // Nettoyer l'URL - enlever les attributs HTML après l'URL
        $value = trim($value);
        
        // Extraire seulement l'URL (avant le premier espace ou guillemet)
        if (preg_match('/^(https?:\/\/[^\s"\']+)/i', $value, $matches)) {
            $value = $matches[1];
        }
        
        // Vérifier si c'est une URL valide
        if (!filter_var($value, FILTER_VALIDATE_URL)) {
            $fail('يجب أن يكون رابط صحيح.');
            return;
        }
        
        $value = strtolower($value);
        
        // Patterns pour liens Google Maps standards
        $standardPatterns = [
            '/^(https?:\/\/)?(www\.)?google\.[a-z]{2,3}(\.[a-z]{2})?\/maps\//',
            '/^(https?:\/\/)?(maps\.google\.[a-z]{2,3}(\.[a-z]{2})?\/)/',
            '/^(https?:\/\/)?goo\.gl\/maps\//',
            '/^(https?:\/\/)?maps\.app\.goo\.gl\//',
            '/^(https?:\/\/)?(www\.)?google\.[a-z]{2,3}(\.[a-z]{2})?\/maps\?q=/',
            '/^(https?:\/\/)?(www\.)?google\.[a-z]{2,3}(\.[a-z]{2})?\/maps\/dir\//',
        ];
        
        // Pattern pour liens embed
        $embedPattern = '/^(https?:\/\/)?(www\.)?(maps\.)?google\.[a-z]{2,3}(\.[a-z]{2})?\/maps\/embed/';
        
        $isValid = false;
        
        // Vérifier les patterns standards
        foreach ($standardPatterns as $pattern) {
            if (preg_match($pattern, $value)) {
                $isValid = true;
                break;
            }
        }
        
        // Vérifier le pattern embed
        if (!$isValid && preg_match($embedPattern, $value)) {
            $isValid = true;
        }
        
        // Vérification par mots-clés
        if (!$isValid) {
            $keywords = [
                'google.com/maps',
                'maps.google.com',
                'goo.gl/maps',
                'maps.app.goo.gl',
                'google.fr/maps',
                'google.tn/maps',
                'google.co.uk/maps',
                'google.de/maps',
                'google.es/maps',
                'google.it/maps',
                'google.ma/maps',
            ];
            
            foreach ($keywords as $keyword) {
                if (strpos($value, $keyword) !== false) {
                    $isValid = true;
                    break;
                }
            }
        }
        
        if (!$isValid) {
            $fail('يجب أن يكون الرابط من Google Maps. مثال: https://www.google.com/maps/place/... أو https://www.google.com/maps/embed?...');
        }
    },
],
	
        'tarifs' => 'nullable|array|min:1',
        'tarifs.*.type' => 'required|in:groupe,single',
        'tarifs.*.id_type_tarif' => 'required|integer|exists:tp_type_tarif,id_type',
        'tarifs.*.tarif_chambre' => 'nullable|numeric|min:0',
        'tarifs.*.tarif_tente' => 'nullable|numeric|min:0',
        'tarifs.*.remarque' => 'nullable|string|max:150',

    ],[
        'nom_ar.required' => 'الرجاء إدخال الاسم بالعربية.',
        'nom_fr.required' => 'الرجاء إدخال الاسم بالفرنسية.',
        'nom_en.required' => 'الرجاء إدخال الاسم بالإنجليزية.',
'map_link.required' => 'رابط موقع المركز على الخريطة مطلوب.',
    'map_link.max' => 'رابط موقع المركز يجب ألا يتجاوز 200 حرف.',
    'map_link.url' => 'يجب أن يكون رابط صحيح.',
    'map_link.regex' => 'يجب أن يكون الرابط من Google Maps.',
        'contact_tel.required' => 'الرجاء إدخال رقم الهاتف.',
        'fax.required' => 'الرجاء إدخال رقم الفاكس .',
        'contact_email.required' => 'الرجاء إدخال البريد الإلكتروني.',

        'gouvernorat_id.required' => 'الرجاء اختيار الولاية.',
        'delegation_id.required' => 'الرجاء اختيار المعتمدية.',

        'epa.required' => 'الرجاء تحديد الحالة (EPA أو Non EPA).',

        'id_milieu_auberge.required' => 'الرجاء اختيار وسط النزل.',

        'adresse.required' => 'الرجاء إدخال العنوان.',
        'code_postal.required' => 'الرجاء إدخال الرمز البريدي.',

        'nom_directeur.required' => 'الرجاء إدخال اسم المدير.',
        'tel_directeur.required' => 'الرجاء إدخال رقم هاتف المدير.',

        'espace_boisson.required' => 'الرجاء تحديد توفر فضاء المشروبات.',
        'piscine.required' => 'الرجاء تحديد توفر المسبح.',

        'capacite_batiment.required' => 'الرجاء إدخال طاقة استيعاب المبنى.',

        'nb_chambres.required' => 'الرجاء إدخال عدد الغرف.',

        'image.required' => 'الرجاء تحميل صورة.',
        'image.image' => 'الملف يجب أن يكون صورة.',
        'image.mimes' => 'يجب أن تكون صيغة الصورة: jpeg أو png أو jpg أو gif أو webp.',
        'image.max' => 'حجم الصورة كبير جدًا، الحد الأقصى 2 ميغابايت.',

        'description_ar.required' => 'الرجاء إدخال الوصف بالعربية.',
        'description_fr.required' => 'الرجاء إدخال الوصف بالفرنسية.',
        'description_en.required' => 'الرجاء إدخال الوصف بالالإنجليزية.',

        'latitude.required' => 'الرجاء إدخال خط العرض.',
        'longitude.required' => 'الرجاء إدخال خط الطول.',

        'chambres.required' => 'الرجاء إضافة غرفة واحدة على الأقل.',
        'chambres.*.type_chambre_id.required' => 'الرجاء اختيار نوع الغرفة.',
        'chambres.*.capacite.required' => 'الرجاء إدخال طاقة الغرفة.',
        'chambres.*.num_chambre.required' => 'الرجاء إدخال رقم الغرفة.',

        'x.required' => 'الرجاء إدخال خط العرض.',
        'y.required' => 'الرجاء إدخال خط الطول.',
    ]);

    // Candidate directories to try for saving uploaded images (ordered).
    $candidates = [
        '/www/mjs/shared_uploads/images',
        '/home/preprov/www/mjs/shared_uploads/images',
        base_path('shared_uploads/images'),
        base_path('../shared_uploads/images'),
        public_path('../../shared_uploads/images'),
    ];

    $usable = null;
    $tried = [];
    $imagePath = null;

    // If an image was uploaded, try to move it to a usable shared folder and set $imagePath
    if ($request->hasFile('image')) {
        $file = $request->file('image');

        foreach ($candidates as $cand) {
            $rp = @realpath($cand);
            $tried[$cand] = [
                'realpath' => $rp ?: null,
                'exists' => @file_exists($cand) ? 'yes' : 'no',
                'is_dir' => @is_dir($cand) ? 'yes' : 'no',
                'is_writable' => @is_writable($cand) ? 'yes' : 'no'
            ];
            if ($rp && is_dir($rp) && is_writable($rp)) {
                $usable = $rp;
                break;
            }
        }

        if (!$usable) {
            $diag = [
                'cwd' => @getcwd(),
                'open_basedir' => ini_get('open_basedir') ?: '(not set)',
                'tried' => $tried,
                'get_current_user' => @get_current_user(),
            ];
            \Log::warning('No usable upload folder found for auberge image upload (update).', $diag);

            $msg = "Upload folder not accessible by PHP. Checked these candidates: ";
            foreach ($tried as $c => $info) {
                $msg .= "[{$c} -> realpath=" . ($info['realpath'] ?? 'null') . ", exists={$info['exists']}, is_dir={$info['is_dir']}, writable={$info['is_writable']}] ";
            }
            return back()->withInput()->withErrors(['image' => $msg]);
        }

        // sanitize original filename
        $originalName = $file->getClientOriginalName();
        $originalName = basename($originalName);
        $originalName = preg_replace('/[\0\/\\\\<>:"|?*\x00-\x1F]+/u', '-', $originalName);
        if ($originalName === '' || $originalName === '.' || $originalName === '..') {
            return back()->withInput()->withErrors(['image' => 'اسم الملف الأصلي غير صالح.']);
        }

        $targetPath = rtrim($usable, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $originalName;

        // avoid overwriting: append timestamp if exists
        if (file_exists($targetPath)) {
            $dotPos = strrpos($originalName, '.');
            if ($dotPos !== false) {
                $base = substr($originalName, 0, $dotPos);
                $ext  = substr($originalName, $dotPos);
            } else {
                $base = $originalName;
                $ext = '';
            }
            $originalName = $base . '-' . time() . $ext;
            $targetPath = rtrim($usable, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $originalName;
        }

        try {
            $file->move($usable, $originalName);
            $imagePath = $targetPath;
        } catch (\Throwable $e) {
            \Log::error('Auberge image move failed (update): ' . $e->getMessage(), ['exception' => $e, 'usable' => $usable]);
            return back()->withInput()->withErrors(['image' => 'فشل حفظ الصورة على الخادم. (راجع سجلات الخادم).']);
        }
    }

    // Prepare update data (preserve existing image if no new one uploaded)
    $updateData = [
        'nom_ar' => $request->nom_ar,
        'nom_fr' => $request->nom_fr,
        'nom_en' => $request->nom_en,
        'contact_tel' => $request->contact_tel,
        'contact_email' => $request->contact_email,
        'gouver_id' => $request->gouvernorat_id,
        'delegation_id' => $request->delegation_id,
        'epa' => $request->epa,
        'id_milieu_auberge' => $request->id_milieu_auberge,
        'fax' => $request->fax,
        'code_postal' => $request->code_postal,
        'adresse' => $request->adresse,
        'nom_directeur' => $request->nom_directeur,
        'tel_directeur' => $request->tel_directeur,
        'espace_boisson' => $request->has('espace_boisson') ? 1 : 0,
        'piscine' => $request->has('piscine') ? 1 : 0,

        'capacite_batiment' => $request->capacite_batiment,
        'nb_chambres' => $request->nb_chambres,
        'nb_chambres_single' => $request->nb_chambres_single,
        'nb_chambres_double' => $request->nb_chambres_double,
        'nb_chambres_triple' => $request->nb_chambres_triple,
        'nb_chambres_quadra' => $request->nb_chambres_quadra,
        'nb_chambres_plus' => $request->nb_chambres_plus,
        'nb_salles_de_bain' => $request->nb_salles_de_bain,
        'nb_salles_de_bain_comm' => $request->nb_salles_de_bain_comm,
        'nb_chambres_tv' => $request->nb_chambres_tv,
        'nb_chambres_clim' => $request->nb_chambres_clim,
        'nb_chambres_frigo' => $request->nb_chambres_frigo,
        'capacite_tentes' => $request->capacite_tentes,
        'cuisine_fonct' => $request->has('cuisine_fonctionnelle') ? 1 : 0,
        'lieux_culturels' => $request->lieux_culturels,
        'autres_services' => $request->autres_services,
        'description_ar' => $request->description_ar,
        'description_fr' => $request->description_fr,
        'description_en' => $request->description_en,
        'tarif_camping_cars' => $request->tarif_camping_cars,
        'latitude' => $request->latitude,
        'longitude' => $request->longitude,
        'prix_lit'=> $request->prix_lit,
        'x' => $request->x,
        'y' => $request->y,
        'map_link' => $request->map_link,
    ];

    // set image field to new path if uploaded, otherwise leave existing value untouched
    if ($imagePath !== null) {
        $updateData['image'] = $imagePath;
    }

    $auberge->update($updateData);

    // continue with the rest of your existing logic (restaurations, chambres, relations, etc.)
    if ($request->has('restaurations')) {
        foreach ($request->restaurations as $restaurationData) {
            if (!empty($restaurationData['id_rest'])) {
                RestaurationAuberge::where('id_rest', $restaurationData['id_rest'])
                    ->update([
                        'nom_fr' => $restaurationData['nom_fr'],
                        'nom_ar' => $restaurationData['nom_ar'],
                        'nom_en' => $restaurationData['nom_en'],
                        'description_fr' => $restaurationData['description_fr'] ?? null,
                        'description_ar' => $restaurationData['description_ar'] ?? null,
                        'description_en' => $restaurationData['description_en'] ?? null,
                        'tarif' => $restaurationData['tarif'],
                    ]);
            } else {
                RestaurationAuberge::create([
                    'auberge_id' => $auberge->id,
                    'nom_fr' => $restaurationData['nom_fr'],
                    'nom_ar' => $restaurationData['nom_ar'],
                    'nom_en' => $restaurationData['nom_en'],
                    'description_fr' => $restaurationData['description_fr'] ?? null,
                    'description_ar' => $restaurationData['description_ar'] ?? null,
                    'description_en' => $restaurationData['description_en'] ?? null,
                    'tarif' => $restaurationData['tarif'],
                ]);
            }
        }
    }

    if ($request->has('chambres')) {
        foreach ($request->chambres as $ch) {
            if (!empty($ch['id_chambre_tente'])) {
                chambreTente::where('id_chambre_tente', $ch['id_chambre_tente'])
                    ->update([
                        'type_chambre_id' => $ch['type_chambre_id'],
                        'capacite' => $ch['capacite'],
                        'num_chambre' => $ch['num_chambre'],
                    ]);
            } else {
                ChambreTente::create([
                    'auberge_id' => $auberge->id,
                    'type_chambre_id' => $ch['type_chambre_id'],
                    'capacite' => $ch['capacite'],
                    'num_chambre' => $ch['num_chambre'],
                    'type' => 'chambre',
                ]);
            }
        }
    }

    if ($request->has('sports')) {
        $auberge->sports()->sync($request->sports);
    } else {
        $auberge->sports()->sync([]);
    }

    if ($request->has('services')) {
        $auberge->services()->sync($request->services);
    } else {
        $auberge->services()->sync([]);
    }

    if ($request->has('espaces')) {
        foreach ($request->espaces as $espaceData) {
            if (!empty($espaceData['id_espace'])) {
                EspaceAuberge::where('id_espace', $espaceData['id_espace'])->update([
                    'nom_ar' => $espaceData['nom_ar'],
                    'nom_fr' => $espaceData['nom_fr'],
                    'nom_en' => $espaceData['nom_en'],
                    'id_auberge' => $auberge->id,
                ]);
            } else {
                EspaceAuberge::create([
                    'nom_ar' => $espaceData['nom_ar'],
                    'nom_fr' => $espaceData['nom_fr'],
                    'nom_en' => $espaceData['nom_en'],
                    'id_auberge' => $auberge->id,
                ]);
            }
        }
    }

    if ($request->has('terrains')) {
        foreach ($request->terrains as $terrainsData) {
            if (!empty($terrainsData['id_terrain'])) {
                TerrainSportifAuberge::where('id_terrain', $terrainsData['id_terrain'])->update([
                    'nom_ar' => $terrainsData['nom_ar'],
                    'nom_fr' => $terrainsData['nom_fr'],
                    'nom_en' => $terrainsData['nom_en'],
                    'id_auberge' => $auberge->id,
                ]);
            } else {
                TerrainSportifAuberge::create([
                    'nom_ar' => $terrainsData['nom_ar'],
                    'nom_fr' => $terrainsData['nom_fr'],
                    'nom_en' => $terrainsData['nom_en'],
                    'id_auberge' => $auberge->id,
                ]);
            }
        }
    }

    if ($request->has('services_auberge')) {
        foreach ($request->services_auberge as $services_aubergeData) {
            if (!empty($services_aubergeData['id_service'])) {
                ServiceAuberge::where('id_service', $services_aubergeData['id_service'])->update([
                    'nom_ar' => $services_aubergeData['nom_ar'],
                    'nom_fr' => $services_aubergeData['nom_fr'],
                    'nom_en' => $services_aubergeData['nom_en'],
                    'id_auberge' => $auberge->id,
                ]);
            } else {
                ServiceAuberge::create([
                    'nom_ar' => $services_aubergeData['nom_ar'],
                    'nom_fr' => $services_aubergeData['nom_fr'],
                    'nom_en' => $services_aubergeData['nom_en'],
                    'id_auberge' => $auberge->id,
                ]);
            }
        }
    }

    $auberge->tarifs()->delete();

    if ($request->has('tarifs')) {
        foreach ($request->tarifs as $tarifData) {
            TarifAuberge::create([
                'id_auberge' => $auberge->id,
                'type' => $tarifData['type'],
                'id_type_tarif' => $tarifData['id_type_tarif'],
                'tarif_chambre' => $tarifData['tarif_chambre'] ?? 0,
                'tarif_tente' => $tarifData['tarif_tente'] ?? 0,
                'remarque' => $tarifData['remarque'] ?? null,
            ]);
        }
    }

 return redirect()->route('auberges2.show', ['id' => $auberge->id])
				->with('success', 'تم تحديث بيانات ' . $auberge->nom_ar . ' بنجاح');
}

	
/*	public function update2(Request $request, $id)
    {
        $auberge = Auberge::findOrFail($id);
        $request->validate([
            'nom_ar' => 'required|string|max:100',
            'nom_fr' => 'required|string|max:100',
            'nom_en' => 'required|string|max:100',
            'contact_tel' => 'required|string|max:50',
			'fax' => 'required|string|max:12',
            'contact_email' => 'required|email|max:255',
            'gouvernorat_id' => 'required|integer',
            'delegation_id' => 'required|integer',
            'epa' => 'required|string|in:EPA,Non EPA',
            'id_milieu_auberge' => 'required|integer',
            'adresse' => 'required|string|max:200',
			'code_postal' => 'required|string|max:20',
            'nom_directeur' => 'required|string|max:255',
            'tel_directeur' => 'required|string|max:50',
            'espace_boisson' => 'required|boolean',
            'piscine' => 'required|boolean',
            
            'cuisine_fonctionnelle' => 'sometimes|boolean',
            'capacite_batiment' => 'required|integer',
            'nb_chambres' => 'required|integer',
            'nb_chambres_single' => 'nullable|integer',
            'nb_chambres_double' => 'nullable|integer',
            'nb_chambres_triple' => 'nullable|integer',
            'nb_chambres_quadra' => 'nullable|integer',
            'nb_chambres_plus' => 'nullable|integer',
            'nb_salles_de_bain' => 'nullable|integer',
            'nb_salles_de_bain_comm' => 'nullable|integer',
            'nb_chambres_tv' => 'nullable|integer',
            'nb_chambres_clim' => 'nullable|integer',
            'nb_chambres_frigo' => 'nullable|integer',
            'capacite_tentes' => 'nullable|integer',
            'lieux_culturels' => 'nullable|string',
            'autres_services' => 'nullable|string',
            //'image' => 'required|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
			'image' => $auberge->image ? 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048' : 'required|image|mimes:jpeg,png,jpg,gif,webp|max:2048',

            'description_ar' => 'required|string|max:2000',
            'description_fr' => 'required|string|max:2000',
            'description_en' => 'required|string|max:2000',
           // 'latitude' => 'required|string|max:50',
            //'longitude' => 'required|string|max:50',
            'prix_lit' => 'nullable|numeric|min:0',
			'restaurations' => 'nullable|array|min:1',
            'restaurations.*.nom_fr' => 'nullable|string|max:255',
            'restaurations.*.nom_ar' => 'nullable|string|max:255',
            'restaurations.*.nom_en' => 'nullable|string|max:255',
            'restaurations.*.description_fr' => 'nullable|string',
            'restaurations.*.description_ar' => 'nullable|string',
            'restaurations.*.description_en' => 'nullable|string',
            'restaurations.*.tarif' => 'nullable|numeric|min:0',
            'chambres' => 'nullable|array|min:1',
            'chambres.*.type_chambre_id' => 'required|integer|exists:tp_type_chambre,id_type',
            'chambres.*.capacite' => 'required|integer|min:1',
            'chambres.*.num_chambre' => 'required|string|max:11',

            'espaces' => 'nullable|array|min:1',
            'espaces.*.nom_ar' => 'nullable|string|max:150',
            'espaces.*.nom_fr' => 'nullable|string|max:150',
            'espaces.*.nom_en' => 'nullable|string|max:150',

            'terrains' => 'nullable|array|min:1',
            'terrains.*.nom_ar' => 'nullable|string|max:150',
            'terrains.*.nom_fr' => 'nullable|string|max:150',
            'terrains.*.nom_en' => 'nullable|string|max:150',

            'services_auberge' => 'nullable|array|min:1',
            'services_auberge.*.nom_ar' => 'nullable|string|max:150',
            'services_auberge.*.nom_fr' => 'nullable|string|max:150',
            'services_auberge.*.nom_en' => 'nullable|string|max:150',
            
            //'x' => 'required|numeric',
			//'y' => 'required|numeric',


            'latitude' => ['required', 'numeric', 'between:0,99', 'regex:/^\d{1,2}(\.\d{4})?$/'],
            'longitude' => ['required', 'numeric', 'between:0,99', 'regex:/^\d{1,2}(\.\d{4})?$/'],
            
            'x' => ['required', 'numeric', 'between:0,99', 'regex:/^\d{1,2}(\.\d{4})?$/'],
            'y' => ['required', 'numeric', 'between:0,99', 'regex:/^\d{1,2}(\.\d{4})?$/'],
            'tarif_camping_cars' => 'nullable|numeric|min:0',

            'tarifs' => 'nullable|array|min:1',
            'tarifs.*.type' => 'required|in:groupe,single',
            'tarifs.*.id_type_tarif' => 'required|integer|exists:tp_type_tarif,id_type',
            'tarifs.*.tarif_chambre' => 'nullable|numeric|min:0',
            'tarifs.*.tarif_tente' => 'nullable|numeric|min:0',
            'tarifs.*.remarque' => 'nullable|string|max:150',
              
        ],[
			'nom_ar.required' => 'الرجاء إدخال الاسم بالعربية.',
			'nom_fr.required' => 'الرجاء إدخال الاسم بالفرنسية.',
			'nom_en.required' => 'الرجاء إدخال الاسم بالإنجليزية.',

			'contact_tel.required' => 'الرجاء إدخال رقم الهاتف.',
			'fax.required' => 'الرجاء إدخال رقم الفاكس .',
			'contact_email.required' => 'الرجاء إدخال البريد الإلكتروني.',

			'gouvernorat_id.required' => 'الرجاء اختيار الولاية.',
			'delegation_id.required' => 'الرجاء اختيار المعتمدية.',

			'epa.required' => 'الرجاء تحديد الحالة (EPA أو Non EPA).',

			'id_milieu_auberge.required' => 'الرجاء اختيار وسط النزل.',

			'adresse.required' => 'الرجاء إدخال العنوان.',
			'code_postal.required' => 'الرجاء إدخال الرمز البريدي.',

			'nom_directeur.required' => 'الرجاء إدخال اسم المدير.',
			'tel_directeur.required' => 'الرجاء إدخال رقم هاتف المدير.',

			'espace_boisson.required' => 'الرجاء تحديد توفر فضاء المشروبات.',
			'piscine.required' => 'الرجاء تحديد توفر المسبح.',

			'capacite_batiment.required' => 'الرجاء إدخال طاقة استيعاب المبنى.',

			'nb_chambres.required' => 'الرجاء إدخال عدد الغرف.',
		
			'image.required' => 'الرجاء تحميل صورة.',
			'image.image' => 'الملف يجب أن يكون صورة.',
			'image.mimes' => 'يجب أن تكون صيغة الصورة: jpeg أو png أو jpg أو gif أو webp.',
			'image.max' => 'حجم الصورة كبير جدًا، الحد الأقصى 2 ميغابايت.',

			'description_ar.required' => 'الرجاء إدخال الوصف بالعربية.',
			'description_fr.required' => 'الرجاء إدخال الوصف بالفرنسية.',
			'description_en.required' => 'الرجاء إدخال الوصف بالإنجليزية.',

			'latitude.required' => 'الرجاء إدخال خط العرض.',
			'longitude.required' => 'الرجاء إدخال خط الطول.',

			//'prix_lit.required' => 'الرجاء إدخال سعر السرير.',

			
			// Chambres
			'chambres.required' => 'الرجاء إضافة غرفة واحدة على الأقل.',
			'chambres.*.type_chambre_id.required' => 'الرجاء اختيار نوع الغرفة.',
			'chambres.*.capacite.required' => 'الرجاء إدخال طاقة الغرفة.',
			'chambres.*.num_chambre.required' => 'الرجاء إدخال رقم الغرفة.',

			
		]);

        $auberge->update([
            'nom_ar' => $request->nom_ar,
            'nom_fr' => $request->nom_fr,
            'nom_en' => $request->nom_en,
            'contact_tel' => $request->contact_tel,
            'contact_email' => $request->contact_email,
            'gouver_id' => $request->gouvernorat_id,
            'delegation_id' => $request->delegation_id,
            'epa' => $request->epa,
            'id_milieu_auberge' => $request->id_milieu_auberge,
            'fax' => $request->fax,
            'code_postal' => $request->code_postal,
            'adresse' => $request->adresse,
            'nom_directeur' => $request->nom_directeur,
            'tel_directeur' => $request->tel_directeur,
            'espace_boisson' => $request->has('espace_boisson') ? 1 : 0,
            'piscine' => $request->has('piscine') ? 1 : 0,
            
            'capacite_batiment' => $request->capacite_batiment,
            'nb_chambres' => $request->nb_chambres,
            'nb_chambres_single' => $request->nb_chambres_single,
            'nb_chambres_double' => $request->nb_chambres_double,
            'nb_chambres_triple' => $request->nb_chambres_triple,
            'nb_chambres_quadra' => $request->nb_chambres_quadra,
            'nb_chambres_plus' => $request->nb_chambres_plus,
            'nb_salles_de_bain' => $request->nb_salles_de_bain,
            'nb_salles_de_bain_comm' => $request->nb_salles_de_bain_comm,
            'nb_chambres_tv' => $request->nb_chambres_tv,
            'nb_chambres_clim' => $request->nb_chambres_clim,
            'nb_chambres_frigo' => $request->nb_chambres_frigo,
            'capacite_tentes' => $request->capacite_tentes,
            'cuisine_fonct' => $request->has('cuisine_fonctionnelle') ? 1 : 0,
            'lieux_culturels' => $request->lieux_culturels,
            'autres_services' => $request->autres_services,
            'description_ar' => $request->description_ar,
            'description_fr' => $request->description_fr,
            'description_en' => $request->description_en,
            'latitude' => $request->latitude,
            'longitude' => $request->longitude,
            'prix_lit'=> $request->prix_lit,
            'tarif_camping_cars' => $request->tarif_camping_cars,
			'x' => $request->x,               
			'y' => $request->y,               
			'map_link' => $request->map_link,
        ]);

        

        if ($request->hasFile('image')) {
            $imageData = file_get_contents($request->file('image')->getRealPath());
            $auberge->image = $imageData;
            $auberge->save();
        }


        if ($request->has('restaurations')) {
            foreach ($request->restaurations as $restaurationData) {
                if (!empty($restaurationData['id_rest'])) {
                    RestaurationAuberge::where('id_rest', $restaurationData['id_rest'])
                        ->update([
                            'nom_fr' => $restaurationData['nom_fr'],
                            'nom_ar' => $restaurationData['nom_ar'],
                            'nom_en' => $restaurationData['nom_en'],
                            'description_fr' => $restaurationData['description_fr'] ?? null,
                            'description_ar' => $restaurationData['description_ar'] ?? null,
                            'description_en' => $restaurationData['description_en'] ?? null,
                            'tarif' => $restaurationData['tarif'],
                        ]);
                } 
                else {
                    RestaurationAuberge::create([
                        'auberge_id' => $auberge->id,
                        'nom_fr' => $restaurationData['nom_fr'],
                        'nom_ar' => $restaurationData['nom_ar'],
                        'nom_en' => $restaurationData['nom_en'],
                        'description_fr' => $restaurationData['description_fr'] ?? null,
                        'description_ar' => $restaurationData['description_ar'] ?? null,
                        'description_en' => $restaurationData['description_en'] ?? null,
                        'tarif' => $restaurationData['tarif'],
                    ]);
                }

            }
        }

        if ($request->has('chambres')) {

            foreach ($request->chambres as $ch) {

                if (!empty($ch['id_chambre_tente'])) {
                    chambreTente::where('id_chambre_tente', $ch['id_chambre_tente'])
                        ->update([
                            'type_chambre_id' => $ch['type_chambre_id'],
                            'capacite' => $ch['capacite'],
                            'num_chambre' => $ch['num_chambre'],
                        ]);
                }
                else {
                    ChambreTente::create([
                        'auberge_id' => $auberge->id,
                        'type_chambre_id' => $ch['type_chambre_id'],
                        'capacite' => $ch['capacite'],
                        'num_chambre' => $ch['num_chambre'],
                        'type' => 'chambre',
                    ]);
                }
            }
        }


        if ($request->has('sports')) {
            $auberge->sports()->sync($request->sports);
        } else {
            $auberge->sports()->sync([]);
        }

        if ($request->has('services')) {
            $auberge->services()->sync($request->services);
        } else {
            $auberge->services()->sync([]);
        }

            



            if ($request->has('espaces')) {
                foreach ($request->espaces as $espaceData) {
                    if (!empty($espaceData['id_espace'])) {
                        EspaceAuberge::where('id_espace', $espaceData['id_espace'])->update([
                            'nom_ar' => $espaceData['nom_ar'],
                            'nom_fr' => $espaceData['nom_fr'],
                            'nom_en' => $espaceData['nom_en'],
                            'id_auberge' => $auberge->id,
                        ]);
                    } else {
                        EspaceAuberge::create([
                            'nom_ar' => $espaceData['nom_ar'],
                            'nom_fr' => $espaceData['nom_fr'],
                            'nom_en' => $espaceData['nom_en'],
                            'id_auberge' => $auberge->id,
                        ]);
                    }
                }
            }


            if ($request->has('terrains')) {
                foreach ($request->terrains as $terrainsData) {
                    if (!empty($terrainsData['id_terrain'])) {
                        TerrainSportifAuberge::where('id_terrain', $terrainsData['id_terrain'])->update([
                            'nom_ar' => $terrainsData['nom_ar'],
                            'nom_fr' => $terrainsData['nom_fr'],
                            'nom_en' => $terrainsData['nom_en'],
                            'id_auberge' => $auberge->id,
                        ]);
                    } else {
                        TerrainSportifAuberge::create([
                            'nom_ar' => $terrainsData['nom_ar'],
                            'nom_fr' => $terrainsData['nom_fr'],
                            'nom_en' => $terrainsData['nom_en'],
                            'id_auberge' => $auberge->id,
                        ]);
                    }
                }
            }

            if ($request->has('services_auberge')) {
                foreach ($request->services_auberge as $services_aubergeData) {
                    if (!empty($services_aubergeData['id_service'])) {
                        ServiceAuberge::where('id_service', $services_aubergeData['id_service'])->update([
                            'nom_ar' => $services_aubergeData['nom_ar'],
                            'nom_fr' => $services_aubergeData['nom_fr'],
                            'nom_en' => $services_aubergeData['nom_en'],
                            'id_auberge' => $auberge->id,
                        ]);
                    } else {
                        ServiceAuberge::create([
                            'nom_ar' => $services_aubergeData['nom_ar'],
                            'nom_fr' => $services_aubergeData['nom_fr'],
                            'nom_en' => $services_aubergeData['nom_en'],
                            'id_auberge' => $auberge->id,
                        ]);
                    }
                }
            }

             $auberge->tarifs()->delete();

                if ($request->has('tarifs')) {
                    foreach ($request->tarifs as $tarifData) {
                        TarifAuberge::create([
                            'id_auberge' => $auberge->id,
                            'type' => $tarifData['type'],
                            'id_type_tarif' => $tarifData['id_type_tarif'],
                            'tarif_chambre' => $tarifData['tarif_chambre'] ?? 0,
                            'tarif_tente' => $tarifData['tarif_tente'] ?? 0,
                            'remarque' => $tarifData['remarque'] ?? null,
                        ]);
                    }
                }


			return redirect()->route('auberges2.show', ['id' => $auberge->id])
				->with('success', 'تم تحديث بيانات ' . $auberge->nom_ar . ' بنجاح');
		}
*/
    public function show($id)
{
    $auberge = Auberge::with('espaces','terrains','services_auberge')->findOrFail($id);
    $milieux = MilieuAuberge::all();
    $gouvernorats = Gouvernorat::all();
    $delegations = Delegation::all();
    $sports = TpSportsAuberge::all();
    $services = TpServiceAuberge::all();
    $typesChambres = TypeChambre::all();
    $typeTarif = TpTypeTarif::all();

    // compute image source (data:base64 or URL) for the auberge image
    $computeImageSrc = function ($stored) {
        if (empty($stored)) return null;

        // already a full URL?
        if (filter_var($stored, FILTER_VALIDATE_URL)) {
            return $stored;
        }

        // candidate filesystem paths (ordered)
        $serverRoot = '/www/mjs';
        $candidates = [
            $stored, // as stored (maybe absolute path or relative path)
        ];

        // try prefixing server root if stored is a relative path or filename
        if (strpos($stored, $serverRoot) !== 0) {
            $candidates[] = rtrim($serverRoot, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . ltrim($stored, DIRECTORY_SEPARATOR);
        }

        // fallback to shared uploads images folder using basename
        $candidates[] = rtrim($serverRoot, DIRECTORY_SEPARATOR) . '/shared_uploads/images/' . basename($stored);

        // public path fallback (if value is relative to public/)
        $candidates[] = public_path(ltrim($stored, '/'));

        // storage/app fallback
        $candidates[] = storage_path('app/' . ltrim($stored, '/'));

        // dedupe & normalize
        $seen = [];
        $candidates = array_values(array_filter(array_map(function ($p) use (&$seen) {
            $p = (string)$p;
            if ($p === '') return null;
            // strip file:// if present
            $p = preg_replace('#^file://#', '', $p);
            if (isset($seen[$p])) return null;
            $seen[$p] = true;
            return $p;
        }, $candidates)));

        // try to read a real file
        foreach ($candidates as $path) {
            if (@file_exists($path) && @is_file($path) && @is_readable($path)) {
                $mime = @mime_content_type($path) ?: 'image/jpeg';
                $contents = @file_get_contents($path);
                if ($contents !== false && $contents !== null) {
                    return "data:{$mime};base64," . base64_encode($contents);
                }
            }
        }

        // if the stored value begins with slash, expose it as an absolute URL
        if (strpos($stored, '/') === 0) {
            return url(ltrim($stored, '/'));
        }

        // fallback: stored might be raw binary blob — return it base64-encoded (assume jpeg)
        try {
            return 'data:image/jpeg;base64,' . base64_encode($stored);
        } catch (\Throwable $e) {
            return null;
        }
    };

    $imageSrc = $computeImageSrc($auberge->image);

    return view('backoffice.auberges.creation_centre.show', compact(
        'auberge','typeTarif','milieux','gouvernorats','sports','services','delegations','typesChambres','imageSrc'
    ));
}



	
	public function showauberge($id)
	{
		$auberge = Auberge::with(['gouvernorat', 'delegation'])->findOrFail($id);

		return view('backoffice.auberges.creation_centre.show2', compact('auberge'));
	}


    public function destroy($id)
    {
        $auberge = Auberge::findOrFail($id);

        try {
            $auberge->delete();
            return redirect()->route('auberges.index')->with('success', 'تم الحذف بنجاح');
        } catch (\Exception $e) {
            return redirect()->route('auberges.index')->with('error', 'حدث خطأ أثناء الحذف');
        }
    }
	
	
	public function getDelegations_auberge($gouvernoratId)

    {

        $delegations = Delegation::where('gouver_id', $gouvernoratId)->get();

        return response()->json($delegations);

    }


}